package com.elitescloud.cloudt.system.service.repo;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.model.vo.query.org.EmpGroupEmpPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.EmpGroupEmpPageRespVO;
import com.elitescloud.cloudt.system.service.model.entity.QSysEmpGroupDO;
import com.elitescloud.cloudt.system.service.model.entity.QSysEmpGroupEmpDO;
import com.elitescloud.cloudt.system.service.model.entity.QSysEmployeeDO;
import com.elitescloud.cloudt.system.service.model.entity.SysEmpGroupEmpDO;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.JPAExpressions;
import org.springframework.stereotype.Repository;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotEmpty;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2023/2/2
 */
@Repository
public class EmpGroupEmpRepoProc extends BaseRepoProc<SysEmpGroupEmpDO> {
    private static final QSysEmpGroupEmpDO QDO = QSysEmpGroupEmpDO.sysEmpGroupEmpDO;
    private static final QSysEmpGroupDO QDO_GROUP = QSysEmpGroupDO.sysEmpGroupDO;
    private static final QSysEmployeeDO QDO_EMP = QSysEmployeeDO.sysEmployeeDO;

    public EmpGroupEmpRepoProc() {
        super(QDO);
    }

    /**
     * 删除员工组的员工
     *
     * @param groupId 员工组ID
     */
    public void deleteEmployeeByGroupId(long groupId) {
        super.jpaQueryFactory.delete(QDO)
                .where(QDO.groupId.eq(groupId))
                .execute();
    }

    /**
     * 删除员工组的员工
     *
     * @param groupId 员工组ID
     * @param empId   员工ID
     */
    public void deleteEmployeeByGroupId(long groupId, long empId) {
        super.jpaQueryFactory.delete(QDO)
                .where(QDO.groupId.eq(groupId).and(QDO.empId.eq(empId)))
                .execute();
    }

    /**
     * 获取员工组的员工ID
     *
     * @param groupId 员工组ID
     * @return 员工ID列表
     */
    public List<Long> getEmployeeIdOfGroup(long groupId) {
        return jpaQueryFactory.select(QDO.empId)
                .from(QDO)
                .where(QDO.groupId.eq(groupId))
                .fetch();
    }

    /**
     * 获取员工组的员工ID
     *
     * @param groupCode 员工组编号
     * @return 员工ID列表
     */
    public List<Long> getEmployeeIdOfGroup(String groupCode) {
        return jpaQueryFactory.select(QDO.empId)
                .from(QDO)
                .where(QDO.groupId.eq(
                        JPAExpressions.select(QDO_GROUP.id).from(QDO_GROUP).where(QDO_GROUP.code.eq(groupCode))
                ))
                .fetch();
    }

    /**
     * 分页查询员工组下的员工管理
     *
     * @param queryVO 查询参数
     * @return 员工列表
     */
    public PagingVO<EmpGroupEmpPageRespVO> pageMng(EmpGroupEmpPageQueryVO queryVO) {
        if (queryVO.getGroupId() == null) {
            return PagingVO.empty();
        }
        var predicate = PredicateBuilder.builder()
                .andEq(true, QDO.groupId, queryVO.getGroupId())
                .andLike(QDO_EMP.code, queryVO.getEmployeeCode())
                .andLike(QDO_EMP.lastName, queryVO.getEmpName())
                .build();
        var bean = Projections.bean(EmpGroupEmpPageRespVO.class, QDO_EMP.id.as("employeeId"),
                QDO_EMP.code.as("employeeCode"), QDO_EMP.lastName.as("employeeName"),
                QDO_EMP.userId.as("userId"), QDO_EMP.duty.as("duty"),
                QDO_EMP.enabled.as("enabled"), QDO_EMP.served.as("served"),
                QDO.createTime.as("createTime"));
        var jpaQuery = super.jpaQueryFactory.select(bean)
                .from(QDO)
                .leftJoin(QDO_EMP).on(QDO_EMP.id.eq(QDO.empId))
                .where(predicate);
        return super.queryByPage(jpaQuery, queryVO.getPageRequest());
    }

    /**
     * 查询员工组下的用户ID
     *
     * @param groupCode 员工组编号
     * @return 用户ID
     */
    public List<Long> queryUserIdOfGroup(@NotEmpty String groupCode) {
        return jpaQueryFactory.select(QDO_EMP.userId)
                .from(QDO_EMP).where(QDO_EMP.id.in(
                        JPAExpressions.select(QDO.empId)
                                .from(QDO)
                                .where(QDO.groupId.eq(
                                        JPAExpressions.select(QDO_GROUP.id).from(QDO_GROUP).where(QDO_GROUP.code.eq(groupCode))
                                ))
                )).fetch();
    }

    /**
     * 查询员工组下的用户ID
     *
     * @param groupCodes 员工组编号
     * @return 员工组编号与用户ID的map
     */
    public Map<String, List<Long>> queryUserIdOfGroup(@NotEmpty Collection<String> groupCodes) {
        return jpaQueryFactory.select(QDO_EMP.userId, QDO_GROUP.code)
                .from(QDO_EMP)
                .leftJoin(QDO).on(QDO.empId.eq(QDO_EMP.id))
                .leftJoin(QDO_GROUP).on(QDO_GROUP.id.eq(QDO.groupId))
                .where(QDO_GROUP.code.in(groupCodes))
                .fetch()
                .stream()
                .collect(Collectors.groupingBy(t -> t.get(QDO_GROUP.code)
                        , Collectors.collectingAndThen(Collectors.toList(), t -> t.stream().map(tt -> tt.get(QDO_EMP.userId)).collect(Collectors.toList()))));
    }

    /**
     * 查询员工组下的用户ID
     *
     * @param groupCodes 员工组编号
     * @return 员工组编号与用户ID的map
     */
    public Map<String, List<IdCodeNameParam>> queryUserOfGroup(@NotEmpty Collection<String> groupCodes) {
        return jpaQueryFactory.select(QDO_EMP.userId, QDO_EMP.code, QDO_EMP.username, QDO_EMP.lastName, QDO_EMP.firstName, QDO_GROUP.code)
                .from(QDO_EMP)
                .leftJoin(QDO).on(QDO.empId.eq(QDO_EMP.id))
                .leftJoin(QDO_GROUP).on(QDO_GROUP.id.eq(QDO.groupId))
                .where(QDO_GROUP.code.in(groupCodes))
                .fetch()
                .stream()
                .collect(Collectors.groupingBy(t -> t.get(QDO_GROUP.code)
                        , Collectors.mapping(t -> {
                            IdCodeNameParam idCodeNameParam = new IdCodeNameParam();
                            idCodeNameParam.setId(t.get(QDO_EMP.userId));
                            idCodeNameParam.setCode(CharSequenceUtil.blankToDefault(t.get(QDO_EMP.code), t.get(QDO_EMP.username)));
                            idCodeNameParam.setName(getUserFullName(t.get(QDO_EMP.lastName), t.get(QDO_EMP.firstName)));

                            return idCodeNameParam;
                        }, Collectors.toList())));
    }

    /**
     * 获取与指定员工同员工组的员工ID
     *
     * @param empId 员工ID
     * @return 员工ID
     */
    public List<Long> queryEmpIdOfSameGroup(long empId) {
        QSysEmpGroupEmpDO qdoSub = new QSysEmpGroupEmpDO("qdoSub");
        var groupSubQuery = JPAExpressions.select(QDO_GROUP.id)
                .from(QDO_GROUP)
                .where(QDO_GROUP.id.in(
                        JPAExpressions.select(qdoSub.groupId).from(qdoSub).where(qdoSub.empId.eq(empId))
                ).and(QDO_GROUP.enabled.eq(true)));

        return jpaQueryFactory.select(QDO.empId)
                .from(QDO)
                .where(QDO.groupId.in(groupSubQuery))
                .fetch();
    }

    /**
     * 获取与指定员工同员工组的员工账号ID
     *
     * @param empId 员工ID
     * @return 员工ID
     */
    public List<Long> queryEmpUserIdOfSameGroup(long empId) {
        QSysEmpGroupEmpDO qdoSub = new QSysEmpGroupEmpDO("qdoSub");
        var groupSubQuery = JPAExpressions.select(QDO_GROUP.id)
                .from(QDO_GROUP)
                .where(QDO_GROUP.id.in(
                        JPAExpressions.select(qdoSub.groupId).from(qdoSub).where(qdoSub.empId.eq(empId))
                ).and(QDO_GROUP.enabled.eq(true)));

        return jpaQueryFactory.select(QDO_EMP.userId)
                .from(QDO_EMP)
                .where(QDO_EMP.id.in(
                        JPAExpressions.select(QDO.empId)
                                .from(QDO)
                                .where(QDO.groupId.in(groupSubQuery))
                ).and(QDO_EMP.userId.isNotNull())).fetch()
                .stream()
                .distinct()
                .collect(Collectors.toList());
    }

    /**
     * 获取全名
     *
     * @return 全名
     */
    protected String getUserFullName(String lastName, String firstName) {
        if (!StringUtils.hasText(lastName)) {
            return firstName;
        }

        return lastName + CharSequenceUtil.blankToDefault(firstName, "");
    }
}
