package com.elitescloud.cloudt.system.service.repo;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.system.dto.SysOuBasicDTO;
import com.elitescloud.cloudt.system.model.vo.query.common.CommonCompanyPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.query.org.OuPageQueryVO;
import com.elitescloud.cloudt.system.service.model.entity.QSysOuDO;
import com.elitescloud.cloudt.system.service.model.entity.SysOuDO;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.QBean;
import org.springframework.stereotype.Repository;

import javax.validation.constraints.NotBlank;
import java.util.Collection;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2023/2/1
 */
@Repository
public class OuRepoProc extends BaseRepoProc<SysOuDO> {
    private static final QSysOuDO QDO = QSysOuDO.sysOuDO;

    public OuRepoProc() {
        super(QDO);
    }

    /**
     * 更新启用状态
     *
     * @param id      ID
     * @param enabled 启用状态
     */
    public void updateEnabled(long id, Boolean enabled) {
        super.updateValue(QDO.enabled, enabled, id);
    }

    /**
     * 判断公司编码是否已存在
     *
     * @param ouCode 税率编码
     * @param id     ID
     * @return 是佛已存在
     */
    public boolean existsOuCode(@NotBlank String ouCode, Long id) {
        return super.exists(QDO.ouCode, ouCode, id);
    }

    /**
     * 获取启用状态
     *
     * @param id id
     * @return 启用状态
     */
    public Boolean getEnabled(long id) {
        return super.getValue(QDO.enabled, id);
    }

    /**
     * 获取公司名称
     *
     * @param id id
     * @return 启用状态
     */
    public String getOuName(long id) {
        return super.getValue(QDO.ouName, id);
    }

    /**
     * 获取公司名称
     *
     * @param ids ID
     * @return id与公司名称
     */
    public Map<Long, String> getNameMapById(@NotBlank Collection<Long> ids) {
        return super.jpaQueryFactory.select(QDO.id, QDO.ouName)
                .from(QDO)
                .where(QDO.id.in(ids))
                .fetch()
                .stream()
                .collect(Collectors.toMap(t -> t.get(QDO.id), t -> t.get(QDO.ouName), (t1, t2) -> t1));
    }

    /**
     * 获取公司名称
     *
     * @param codes 公司编码
     * @return code与公司名称
     */
    public Map<String, String> getNameMapByOuCode(@NotBlank Collection<String> codes) {
        return super.jpaQueryFactory.select(QDO.ouCode, QDO.ouName)
                .from(QDO)
                .where(QDO.ouCode.in(codes))
                .fetch()
                .stream()
                .collect(Collectors.toMap(t -> t.get(QDO.ouCode), t -> t.get(QDO.ouName), (t1, t2) -> t1));
    }

    /**
     * 分页查询管理
     *
     * @param queryVO 查询参数
     * @return 公司列表
     */
    public PagingVO<SysOuDO> pageMng(OuPageQueryVO queryVO) {
        var predicate = PredicateBuilder.builder()
                .andLike(QDO.ouCode, queryVO.getOuCode())
                .andLike(QDO.ouName, queryVO.getOuName())
                .andEq(QDO.ouType, queryVO.getOuType())
                .andEq(queryVO.getEnabled() != null, QDO.enabled, queryVO.getEnabled())
                .build();
        return super.queryByPage(predicate, queryVO.getPageRequest());
    }

    /**
     * 分页查询管理
     *
     * @param queryVO 查询参数
     * @return 公司列表
     */
    public PagingVO<SysOuDO> pageQuery(CommonCompanyPageQueryVO queryVO) {
        var predicate = PredicateBuilder.builder()
                .andLike(QDO.ouCode, queryVO.getOuCode())
                .andLike(QDO.ouName, queryVO.getOuName())
                .andEq(QDO.ouType, queryVO.getOuType())
                .andEq(true, QDO.enabled, true)
                .build();
        return super.queryByPage(predicate, queryVO.getPageRequest());
    }

    /**
     * 获取公司信息
     *
     * @param id 公司ID
     * @return 公司信息
     */
    public SysOuBasicDTO getOuBasic(long id) {
        return super.jpaQueryFactory.select(this.qBeanOuBasicDTO())
                .from(QDO)
                .where(QDO.id.eq(id))
                .limit(1)
                .fetchOne();
    }

    private QBean<SysOuBasicDTO> qBeanOuBasicDTO() {
        return Projections.bean(SysOuBasicDTO.class, QDO.id, QDO.ouCode, QDO.ouName, QDO.pinyin, QDO.ouAbbr, QDO.ouType, QDO.enabled);
    }
}
