package com.elitescloud.cloudt.system.service.repo;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.dto.req.SysTaxRateQueryDTO;
import com.elitescloud.cloudt.system.model.vo.query.extend.TaxRatePageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.common.CommonTaxRateRespVO;
import com.elitescloud.cloudt.system.service.model.entity.QSysTaxRateDO;
import com.elitescloud.cloudt.system.service.model.entity.SysTaxRateDO;
import com.querydsl.core.types.Projections;
import org.springframework.stereotype.Repository;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotBlank;
import java.time.LocalDateTime;
import java.util.List;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2023/2/1
 */
@Repository
public class TaxRateRepoProc extends BaseRepoProc<SysTaxRateDO> {
    private static final QSysTaxRateDO QDO = QSysTaxRateDO.sysTaxRateDO;

    public TaxRateRepoProc() {
        super(QDO);
    }

    /**
     * 更新启用状态
     *
     * @param id      ID
     * @param enabled 启用状态
     */
    public void updateEnabled(long id, Boolean enabled) {
        super.updateValue(QDO.enabled, enabled, id);
    }

    /**
     * 判断税率编码是否已存在
     *
     * @param rateNo 税率编码
     * @param id     ID
     * @return 是佛已存在
     */
    public boolean existsRateNo(String rateNo, Long id) {
        return super.exists(QDO.taxRateNo, rateNo, id);
    }

    /**
     * 获取启用状态
     *
     * @param id id
     * @return 启用状态
     */
    public Boolean getEnabled(long id) {
        return super.getValue(QDO.enabled, id);
    }

    /**
     * 根据税率编码获取税率
     *
     * @param taxRateNo   税率编码
     * @param taxRateType 税率类型
     * @return 税率信息
     */
    public SysTaxRateDO getByTaxRateNo(@NotBlank String taxRateNo, String taxRateType) {
        var predicate = QDO.taxRateNo.eq(taxRateNo);
        if (CharSequenceUtil.isNotBlank(taxRateType)) {
            predicate = predicate.and(QDO.taxRateType.eq(taxRateType).or(QDO.taxRateType.isNull()));
        }

        return super.jpaQueryFactory.selectFrom(QDO)
                .where(predicate)
                .limit(1)
                .fetchOne();
    }

    /**
     * 分页查询管理
     *
     * @param queryVO 查询参数
     * @return 税率列表
     */
    public PagingVO<SysTaxRateDO> pageMng(TaxRatePageQueryVO queryVO) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.taxRateType, queryVO.getTaxRateType())
                .andLike(StringUtils.hasText(queryVO.getTaxRateNo()), QDO.taxRateNo, queryVO.getTaxRateNo())
                .andLike(StringUtils.hasText(queryVO.getTaxRateDesc()), QDO.taxRateDesc, queryVO.getTaxRateDesc())
                .andEq(queryVO.getEnabled() != null, QDO.enabled, queryVO.getEnabled())
                .andAfter(queryVO.getValidFromS() != null, QDO.validFrom, queryVO.getValidFromS())
                .andBefore(queryVO.getValidFromE() != null, QDO.validFrom, queryVO.getValidFromE())
                .andAfter(queryVO.getValidToS() != null, QDO.validTo, queryVO.getValidToS())
                .andBefore(queryVO.getValidToE() != null, QDO.validTo, queryVO.getValidToE())
                .build();
        return super.queryByPage(predicate, queryVO.getPageRequest(), QDO.createTime.desc());
    }

    /**
     * 查询有效税率
     *
     * @return 税率列表
     */
    public List<CommonTaxRateRespVO> queryList(String taxRateNo, String taxRateType) {
        var now = LocalDateTime.now();
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.taxRateNo, taxRateNo)
                .andEq(QDO.taxRateType, taxRateType)
                .and(true, () -> QDO.validFrom.loe(now))
                .and(true, () -> QDO.validTo.goe(now))
                .andEq(QDO.enabled, true)
                .build();

        return super.jpaQueryFactory.select(Projections.bean(CommonTaxRateRespVO.class, QDO.taxRateNo, QDO.taxRateType, QDO.taxRateValue, QDO.taxRateDesc))
                .from(QDO)
                .where(predicate)
                .fetch();
    }

    /**
     * 查询税率列表
     *
     * @param queryDTO 查询参数
     * @return 税率列表
     */
    public List<SysTaxRateDO> queryList(SysTaxRateQueryDTO queryDTO) {
        var predicate = PredicateBuilder.builder()
                .and(StringUtils.hasText(queryDTO.getTaxRateType()), () -> QDO.taxRateType.eq(queryDTO.getTaxRateType()).or(QDO.taxRateType.isNull()))
                .andIn(QDO.taxRateNo, queryDTO.getTaxRateNos())
                .andIn(QDO.taxRateValue, queryDTO.getTaxRates())
                .build();
        return super.getList(predicate);
    }
}
