package com.elitescloud.cloudt.system.service.repo;

import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.boot.util.JSONUtil;
import com.elitescloud.cloudt.system.service.common.constant.ThirdPartAccountType;
import com.elitescloud.cloudt.system.service.model.bo.ThirdPartAccountBO;
import com.elitescloud.cloudt.system.service.model.entity.QSysThirdPartyAccountDO;
import com.elitescloud.cloudt.system.service.model.entity.SysThirdPartyAccountDO;
import com.fasterxml.jackson.core.type.TypeReference;
import org.springframework.stereotype.Repository;

import javax.validation.constraints.NotBlank;
import java.util.Map;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2/10/2023
 */
@Repository
public class ThirdPartyAccountRepoProc extends BaseRepoProc<SysThirdPartyAccountDO> {
    private static final QSysThirdPartyAccountDO QDO = QSysThirdPartyAccountDO.sysThirdPartyAccountDO;

    public ThirdPartyAccountRepoProc() {
        super(QDO);
    }

    /**
     * 更新CasId
     *
     * @param id    id
     * @param casId casId
     */
    public void updateCasId(long id, Long casId) {
        super.updateValue(QDO.casId, casId, id);
    }

    /**
     * 获取CasId
     *
     * @param id id
     * @return CasId
     */
    public Long getCasId(long id) {
        return super.getValue(QDO.casId, id);
    }

    /**
     * 根据账号获取租户ID
     *
     * @param account 账号
     * @return CasId
     */
    public Long getTenantIdByAccount(@NotBlank String account) {
        return super.getValueByValue(QDO.sysTenantId, QDO.account, account);
    }

    /**
     * 根据类型获取
     *
     * @param accountType 账号类型
     * @return 配置
     */
    public SysThirdPartyAccountDO getOneByType(@NotBlank ThirdPartAccountType accountType, @NotBlank String businessType, long tenantId) {
        return super.jpaQueryFactory.selectFrom(QDO)
                .where(QDO.sysTenantId.eq(tenantId).and(QDO.accountType.eq(accountType.getValue())).and(QDO.businessType.eq(businessType)))
                .limit(1)
                .fetchOne();
    }

    /**
     * 获取账号ID
     *
     * @param accountType  账号类型
     * @param businessType 业务类型
     * @param tenantId     租户ID
     * @return 账号ID
     */
    public Long getIdByType(@NotBlank ThirdPartAccountType accountType, @NotBlank String businessType, long tenantId) {
        return super.jpaQueryFactory.select(QDO.id)
                .from(QDO)
                .where(QDO.sysTenantId.eq(tenantId).and(QDO.accountType.eq(accountType.getValue())).and(QDO.businessType.eq(businessType)))
                .limit(1)
                .fetchOne();
    }

    /**
     * 判断账号是否已存在
     *
     * @param account 账号
     * @return 是否已存在
     */
    public boolean existsAccount(String account) {
        return super.exists(QDO.account, account);
    }

    /**
     * 获取配置
     *
     * @param accountType
     * @param businessType
     * @param tenantId
     * @return
     */
    public ThirdPartAccountBO getAccountConfig(@NotBlank ThirdPartAccountType accountType, @NotBlank String businessType, long tenantId) {
        var result = super.jpaQueryFactory.select(QDO.sysTenantId, QDO.businessType, QDO.account, QDO.accountType, QDO.configJson)
                .from(QDO)
                .where(QDO.accountType.eq(accountType.getValue()).and(QDO.businessType.eq(businessType)).and(QDO.sysTenantId.eq(tenantId))
                        .and(QDO.enabled.eq(true)))
                .limit(1)
                .fetchOne();
        if (result == null) {
            return null;
        }

        ThirdPartAccountBO accountBO = new ThirdPartAccountBO();
        accountBO.setSysTenantId(result.get(QDO.sysTenantId));
        accountBO.setBusinessType(result.get(QDO.businessType));
        accountBO.setAccount(result.get(QDO.account));
        accountBO.setAccountType(result.get(QDO.accountType));
        accountBO.setConfig(JSONUtil.json2Obj(result.get(QDO.configJson), new TypeReference<>() {
        }));

        return accountBO;
    }
}
