package com.elitescloud.cloudt.system.service.repo;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import com.elitescloud.cloudt.system.model.vo.query.user.sync.UserSyncRecordPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.user.sync.UserSyncRecordDetailRespVO;
import com.elitescloud.cloudt.system.provider.usersync.SyncUserResult;
import com.elitescloud.cloudt.system.service.model.entity.QSysTenantUserDO;
import com.elitescloud.cloudt.system.service.model.entity.QSysUserDO;
import com.elitescloud.cloudt.system.service.model.entity.QSysUserSyncDO;
import com.elitescloud.cloudt.system.service.model.entity.SysUserSyncDO;
import com.querydsl.core.types.Expression;
import com.querydsl.core.types.dsl.StringExpression;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.time.LocalDateTime;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/7/30
 */
@Repository
public class UserSyncRepoProc extends BaseRepoProc<SysUserSyncDO> {
    private static final QSysUserSyncDO QDO = QSysUserSyncDO.sysUserSyncDO;
    private static final QSysUserDO QDO_USER = QSysUserDO.sysUserDO;
    private static final QSysTenantUserDO QDO_TENANT_USER = QSysTenantUserDO.sysTenantUserDO;

    @Autowired
    private TenantClientProvider tenantClientProvider;

    public UserSyncRepoProc() {
        super(QDO);
    }

    /**
     * 更新同步结果
     *
     * @param id
     * @param syncResult
     * @param failReasonDetail
     */
    public void updateSyncResult(long id, SyncUserResult syncResult, String failReasonDetail) {
        super.jpaQueryFactory.update(QDO)
                .set(QDO.syncSuccess, syncResult.isSuccess())
                .set(QDO.syncFinishTime, LocalDateTime.now())
                .set(QDO.failReason, syncResult.getFailMsg())
                .set(QDO.failReasonDetail, failReasonDetail)
                .set(QDO.receipt, syncResult.getReceipt())
                .set(QDO.outerUserId, syncResult.getOuterUserId())
                .set(QDO.outerUserCode, syncResult.getOuterUserCode())
                .where(QDO.id.eq(id))
                .execute();
    }

    /**
     * 获取同步记录
     *
     * @param userId
     * @param outerSysCode
     * @return
     */
    public SysUserSyncDO getUserSync(long userId, @NotBlank String outerSysCode) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.userId, userId)
                .andEq(QDO.outerSysCode, outerSysCode)
                .build();
        return super.getOne(predicate);
    }

    /**
     * 分页查询同步信息记录
     *
     * @param queryVO
     * @param outerSysCode
     * @return
     */
    public PagingVO<UserSyncRecordDetailRespVO> pageMng(@NotNull UserSyncRecordPageQueryVO queryVO, @NotBlank String outerSysCode) {
        var fields = new Expression[]{QDO_USER.id, QDO_USER.username,
                QDO_USER.lastName, QDO_USER.firstName, QDO_USER.mobile, QDO_USER.email, QDO_USER.idCard,
                QDO.outerUserCode, QDO.id, QDO.syncTime, QDO.syncSuccess, QDO.failReason, QDO.syncData};

        var jpaQuery = jpaQueryFactory.select(fields)
                .from(QDO_USER)
                .leftJoin(QDO).on(QDO.userId.eq(QDO_USER.id).and(QDO.outerSysCode.eq(outerSysCode)));

        // 租户
        SysTenantDTO tenant = tenantClientProvider.getCurrentTenant();
        if (tenant != null) {
            // 如果限制了租户，则使用租户过滤
            jpaQuery.leftJoin(QDO_TENANT_USER)
                    .on(QDO_TENANT_USER.sysUserId.eq(QDO_USER.id).and(QDO_TENANT_USER.sysTenantId.eq(tenant.getId())));
        }

        // 查询条件
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.outerUserCode, queryVO.getOuterUserCode())
                .andEq(QDO_USER.username, queryVO.getUsername())
                .andEq(QDO_USER.mobile, queryVO.getMobile())
                .andEq(QDO_USER.email, queryVO.getEmail())
                .andEq(QDO_TENANT_USER.sysTenantId, tenant == null ? null : tenant.getId())
                .andLike(new StringExpression[]{QDO_USER.lastName, QDO_USER.firstName}, queryVO.getFullName())
                .andLike(new StringExpression[]{QDO_USER.username, QDO_USER.lastName, QDO_USER.firstName, QDO.outerUserCode}, queryVO.getKeyword())
                .build();
        jpaQuery.where(predicate);

        // 分页查询
        return super.queryByPage(jpaQuery, queryVO.getPageRequest(), QDO_USER.createTime.asc(), t -> {
            UserSyncRecordDetailRespVO respVO = new UserSyncRecordDetailRespVO();
            respVO.setId(t.get(QDO_USER.id));
            respVO.setUsername(t.get(QDO_USER.username));
            respVO.setFullName(this.getFullName(t.get(QDO_USER.lastName), t.get(QDO_USER.firstName)));
            respVO.setMobile(t.get(QDO_USER.mobile));
            respVO.setEmail(t.get(QDO_USER.email));
            respVO.setIdCard(t.get(QDO_USER.idCard));
            respVO.setOuterUserCode(t.get(QDO.outerUserCode));
            respVO.setSync(t.get(QDO.id) != null && Boolean.TRUE.equals(t.get(QDO.syncSuccess)));
            respVO.setSyncSuccess(t.get(QDO.syncSuccess));
            respVO.setSyncTime(t.get(QDO.syncTime));
            respVO.setSyncData(t.get(QDO.syncData));
            respVO.setFailReason(t.get(QDO.failReason));

            return respVO;
        });
    }

    /**
     * 获取外部账号
     *
     * @param userId
     * @param outerSysCode
     * @return
     */
    public String getOuterUserCodeByUserId(long userId, @NotBlank String outerSysCode) {
        return super.getValue(QDO.outerUserCode, QDO.userId.eq(userId).and(QDO.outerSysCode.eq(outerSysCode)));
    }

    private String getFullName(String lastName, String firstName) {
        if (!StringUtils.hasText(lastName)) {
            return firstName;
        }

        return lastName + CharSequenceUtil.blankToDefault(firstName, "");
    }

}
