package com.elitescloud.cloudt.system.util;

import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.StreamUtils;

import javax.servlet.ReadListener;
import javax.servlet.ServletInputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import java.io.*;
import java.nio.charset.StandardCharsets;

/**
 * @Auther: Mark
 * @Date: 2024/12/17 17:41
 * @Description:
 */
@Slf4j
public class RequestWrapper extends HttpServletRequestWrapper {

    private final byte[] body;

    public RequestWrapper(HttpServletRequest request) {
        super(request);
        String bodyString = getBodyString(request);
        body = bodyString.getBytes(StandardCharsets.UTF_8);
    }

    /**
     * 在使用 @RequestBody 注解的时候
     * 其实框架是调用了 getInputStream() 方法，所以我们要重写这个方法；
     */
    @Override
    public ServletInputStream getInputStream() {
        final ByteArrayInputStream ins = new ByteArrayInputStream(body);
        return new ServletInputStream() {
            @Override
            public int read() {
                return ins.read();
            }

            @Override
            public boolean isFinished() {
                return false;
            }

            @Override
            public boolean isReady() {
                return false;
            }

            @Override
            public void setReadListener(ReadListener readListener) {
            }
        };
    }

    /**
     * 读取 Request Body 的内容信息
     */
    @SneakyThrows
    public String getBodyString() {
        byte[] bodyBytes = StreamUtils.copyToByteArray(this.getInputStream());
        return new String(bodyBytes, getRequest().getCharacterEncoding());
    }

    /**
     * 读取 Request Body 的内容信息
     *
     * @param request The portlet request to be parsed.
     * @return body参数
     */
    private String getBodyString(HttpServletRequest request) {
        StringBuilder sb = new StringBuilder();
        InputStream inputStream = null;
        BufferedReader reader = null;
        try {
            inputStream = request.getInputStream();
            reader = new BufferedReader(new InputStreamReader(inputStream, StandardCharsets.UTF_8));

            char[] bodyCharBuffer = new char[1024];
            int len;
            while ((len = reader.read(bodyCharBuffer)) != -1) {
                sb.append(new String(bodyCharBuffer, 0, len));
            }
        } catch (IOException e) {
//            e.printStackTrace();
            log.warn("[PHOENIX-HTTP] RequestWrapper Stream is closed.");
        } finally {
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
            if (reader != null) {
                try {
                    reader.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return sb.toString();
    }
}
