package com.elitescloud.cloudt.system.util;

import cn.hutool.core.lang.Assert;
import cn.hutool.json.JSONUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.system.service.ToolContextSpi;
//import com.lzhpo.tracer.util.TracerUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.VelocityEngine;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.StringWriter;

/**
 * @Auther: Mark
 * @Date: 2024/12/17 18:58
 * @Description:
 */
@Slf4j
public class TransformUtil {
    private static final Logger logger = LoggerFactory.getLogger(TransformUtil.class);
    private static VelocityEngine velocityEngine;

    private static VelocityEngine getInstance() {
        if (velocityEngine == null) {
            velocityEngine = new VelocityEngine();
            velocityEngine.init();
        }
        return velocityEngine;
    }

    /**
     * 使用模板
     *
     * @param context  上下文
     * @param logTag   日志标签，报错排查问题时需要
     * @param inString 模板
     * @return 解析后数据
     */
    public static String parse(VelocityContext context, String logTag, String inString) {
//        context.put("traceId", TracerUtils.getTraceId());
        // 日期格式化
        context.put("date", new DateFormat());
        // 对象工具
        context.put("bean", new BeanFormat());
        // 注册自定义工具对象
        if (SpringUtils.isPresent(ToolContextSpi.class)) {
            ToolContextSpi toolContext = SpringUtils.getBean(ToolContextSpi.class);
            toolContext.register(context);
        }
        StringWriter out = new StringWriter();
        boolean result = getInstance().evaluate(context, out, logTag, inString);
        if (!result) {
            logger.error("[PHOENIX][ETL] transform error: {}, {}, {}", logTag, inString, context);
            throw new BusinessException("transform error.");
        }
        String outString = out.toString();
        log.info("[PHOENIX][ETL] 转换后报文: ", outString);
        return outString;
    }

    /**
     * 数据解析后，转换为 JSON 字符串
     *
     * @since 0.1.3-SNAPSHOT
     */
    public static String parseToJsonStr(VelocityContext context, String logTag, String inString) {
        return JSONUtil.toJsonPrettyStr(parse(context, logTag, inString));
    }

    /**
     * 数据解析映射
     *
     * @param data     原数据报文
     * @param inString 模板
     * @return 解析后数据
     * @since 0.1.3-SNAPSHOT
     */
    public static String parse(Object data, String inString) {
        return parseToDefault(data, inString, "");
    }

    /**
     * 解析数据，会抛出异常
     *
     * @since 0.1.3-SNAPSHOT
     */
    public static String parseOrThrow(Object data, String inString) {
        Assert.notNull(data, "[PHOENIX][ETL] 数据不能为空");
        Assert.notNull(inString, "[PHOENIX][ETL] 模板不能为空");
        VelocityContext context = new VelocityContext();
        context.put("ctx", data);
        return TransformUtil.parse(context, "", inString);
    }

    /**
     * 解析数据，抛出异常时 返回默认值
     *
     * @since 0.1.3-SNAPSHOT
     */
    public static String parseToDefault(Object data, String inString, String defaultValue) {
        try {
            return parseOrThrow(data, inString);
        } catch (IllegalArgumentException e) {
            logger.error("[PHOENIX][ETL] transform IllegalArgument: {}, {}", inString, data);
        } catch (Exception e) {
            logger.error("[PHOENIX][ETL] transform error: {}, {}", inString, data, e);
        }
        return defaultValue;
    }
}
