package com.elitescloud.cloudt.ucenter.service.impl;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.annotation.SysCodeProc;
import com.elitescloud.cloudt.common.base.ApiCode;
import com.elitescloud.cloudt.common.base.PagingVO;

import com.elitescloud.cloudt.ucenter.api.dto.WebsiteBarDTO;
import com.elitescloud.cloudt.ucenter.api.vo.param.WebsiteBarPagingParam;
import com.elitescloud.cloudt.ucenter.api.vo.param.WebsiteBarShowParam;
import com.elitescloud.cloudt.ucenter.api.vo.param.WebsiteBarSortParam;
import com.elitescloud.cloudt.ucenter.api.vo.resp.UpBarRespVO;
import com.elitescloud.cloudt.ucenter.api.vo.resp.WebsiteBarRespVO;
import com.elitescloud.cloudt.ucenter.api.vo.save.WebsiteBarSaveVO;
import com.elitescloud.cloudt.ucenter.common.constant.ConstantsUcenter;
import com.elitescloud.cloudt.ucenter.common.constant.UdcEnum;
import com.elitescloud.cloudt.ucenter.convert.WebsiteBarConvert;
import com.elitescloud.cloudt.ucenter.entity.WebsiteBarDO;
import com.elitescloud.cloudt.ucenter.repo.WebsiteBarRepo;
import com.elitescloud.cloudt.ucenter.repo.WebsiteBarRepoProc;
import com.elitescloud.cloudt.ucenter.service.WebsiteBarService;
import com.elitescloud.cloudt.ucenter.utils.SysGenerator;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 网站栏目配置
 *
 * @author chaofeng.xia
 * @date 2023/5/26 16:35
 */

@Service
@RequiredArgsConstructor
@Slf4j
public class WebsiteBarServiceImpl implements WebsiteBarService {

    private final WebsiteBarRepo websiteBarRepo;

    private final WebsiteBarRepoProc websiteBarRepoProc;

    private final SysGenerator sysGenerator;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long save(WebsiteBarSaveVO saveVO) {

        // 新增
        if (ObjectUtil.isNull(saveVO.getId())) {
//            checkHomePageFlag(saveVO);
            // 网站导航栏/底栏校验参数
            if (StrUtil.equalsAny(saveVO.getBarType(),
                    UdcEnum.WS_WEBSITE_BAR_TYPE_NAV.getValueCode(), UdcEnum.WS_WEBSITE_BAR_TYPE_BOTTOM.getValueCode())) {
                if (StrUtil.isEmpty(saveVO.getWindowOpenMode())) {
                    throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "打开方式不能为空");
                }
                if (StrUtil.isEmpty(saveVO.getBarType())) {
                    throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "栏位类型不能为空");
                }
            }
            if (!StrUtil.equals(saveVO.getLinkType(), UdcEnum.WS_LINK_TYPE_EXTERNAL.getValueCode())) {
                if (StrUtil.isBlank(saveVO.getTargetType())) {
                    throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "目标类型不能为空");
                }
            }
            String barCode = sysGenerator.generateCode(ConstantsUcenter.BAR_NO);
            if (websiteBarRepo.existsByBarCode(barCode)) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION,
                        "发号器生成的栏目编号：" + barCode + " 已存在，请检查发号器配置是否正确");
            }
            // 使用发号器生成栏目编号
            saveVO.setBarCode(barCode);
        } else {
            // 编辑
            List<WebsiteBarDO> doList = websiteBarRepo.findByPid(saveVO.getId());


            Optional<WebsiteBarDO> byId = websiteBarRepo.findById(saveVO.getId());
            if (byId.isEmpty()) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "没有查询到栏目信息");
            }
            WebsiteBarDO websiteBarDO = byId.get();
            if (ObjectUtil.equals(websiteBarDO.getShowFlag(), Boolean.TRUE)) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "只支持编辑显示状态为否的栏目");
            }
            // 导航栏 并且有 子导航栏
            if (StrUtil.equals(websiteBarDO.getBarType(), UdcEnum.WS_WEBSITE_BAR_TYPE_NAV.getValueCode())
                    && CollectionUtil.isNotEmpty(doList)) {
                // 不允许修改导航栏
                if (!StrUtil.equals(saveVO.getBarType(), UdcEnum.WS_WEBSITE_BAR_TYPE_NAV.getValueCode())) {
                    throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "该栏目下有子导航栏，不能修改位置信息");
                }
                if (ObjectUtil.isNotNull(saveVO.getPid())) {
                    throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "该栏目有子导航栏，不能更改上级栏目");
                }
                // 更新父类名称
                websiteBarRepo.updatePNameByPid(websiteBarDO.getId(), saveVO.getBarName());
            }
            // 编辑的栏目不是首页 校验首页
            if (ObjectUtil.equals(websiteBarDO.getHomePageFlag(), Boolean.FALSE)) {
//                checkHomePageFlag(saveVO);
            }
            saveVO.setPid(websiteBarDO.getPid());
        }
        WebsiteBarDO websiteBarDO = WebsiteBarConvert.INSTANCE.saveVo2Do(saveVO);
        if (ObjectUtil.isNotNull(saveVO.getPid())) {
            Optional<WebsiteBarDO> parentBar = websiteBarRepo.findById(saveVO.getPid());
            if (parentBar.isPresent()) {
                websiteBarDO.setPName(parentBar.get().getBarName());
            }
        }
        WebsiteBarDO save = websiteBarRepo.save(websiteBarDO);
        return save.getId();
    }

    /**
     * PC导航栏只能有一个首页
     *
     * @param saveVO
     */
    private void checkHomePageFlag(WebsiteBarSaveVO saveVO) {
        if (ObjectUtil.equals(saveVO.getHomePageFlag(), Boolean.TRUE)
                && StrUtil.equals(saveVO.getBarType(), UdcEnum.WS_WEBSITE_BAR_TYPE_NAV.getValueCode())) {
            WebsiteBarDO websiteBarDO = websiteBarRepo.findByHomePageFlag(Boolean.TRUE);
            if (ObjectUtil.isNotNull(websiteBarDO)) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "导航栏只能有一个首页");
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Object showUpdate(WebsiteBarShowParam param) {
        websiteBarRepo.updateShowFlagByPid(param.getIds(), param.getShowFlag());
        return websiteBarRepo.updateShowFlagByIdIn(param.getIds(), param.getShowFlag());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Object sortUpdate(WebsiteBarSortParam param) {


        return websiteBarRepo.updateSortNoById(param.getId(), param.getSortNo());
    }

    @Override
    @SysCodeProc
    public WebsiteBarRespVO queryDetail(Long id) {
        Optional<WebsiteBarDO> byId = websiteBarRepo.findById(id);
        if (byId.isEmpty()) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "没有对应的网站栏目详情");
        }
        return WebsiteBarConvert.INSTANCE.do2DetailRespVo(byId.get());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Object batchDeleteByIds(List<Long> ids) {

        if (CollectionUtil.isEmpty(ids)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "删除参数不能为空");
        }
        List<WebsiteBarDO> doList = websiteBarRepo.findByIdIn(ids);
        List<WebsiteBarDO> homePageList = doList.stream().filter(item -> ObjectUtil.equals(item.getHomePageFlag(), Boolean.TRUE)).collect(Collectors.toList());
        if (CollectionUtil.isNotEmpty(homePageList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "不能删除首页栏目");
        }
        List<WebsiteBarDO> showList = doList.stream().filter(item -> ObjectUtil.equals(item.getShowFlag(), Boolean.TRUE)).collect(Collectors.toList());
        if (CollectionUtil.isNotEmpty(showList)) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "只能删除显示状态为否的栏目");
        }

        Integer count = websiteBarRepo.deleteByIdIn(ids);
        // 删除子级
        websiteBarRepo.deleteByParentIdIn(ids);
        return count;
    }

    @Override
    @SysCodeProc
    public PagingVO<WebsiteBarRespVO> search(WebsiteBarPagingParam param) {

        // 1.根据条件查询所有的内容
        List<WebsiteBarDTO> dtoList = websiteBarRepoProc.findAll(param);

        // 分出一级栏目于二级栏目
        List<WebsiteBarDTO> parentDtoList = dtoList.stream().filter(item -> ObjectUtil.isNull(item.getPid())).distinct().collect(Collectors.toList());
        List<WebsiteBarDTO> childDtoList = dtoList.stream().filter(item -> ObjectUtil.isNotNull(item.getPid())).distinct().collect(Collectors.toList());


        // 收集所有的一级栏目id
        Set<Long> parentIdList1 = parentDtoList.stream().map(item -> item.getId()).collect(Collectors.toSet());
        Set<Long> parentIdList2 = childDtoList.stream().map(item -> item.getPid()).collect(Collectors.toSet());
        // 所有的一级id集合
        parentIdList1.addAll(parentIdList2);

        // 查询所有一级栏目
        param.setPidSet(parentIdList1);
        PagingVO<WebsiteBarDTO> parentPage = websiteBarRepoProc.search(param);
        List<WebsiteBarRespVO> parentList = WebsiteBarConvert.INSTANCE.barDTOsToVos(parentPage.getRecords());
        // 设置一级栏目对应的二级栏目
        if (CollectionUtil.isNotEmpty(childDtoList)) {
            List<WebsiteBarRespVO> childList = WebsiteBarConvert.INSTANCE.barDTOsToVos(childDtoList);
            Map<Long, List<WebsiteBarRespVO>> childMap = childList.stream().collect(Collectors.groupingBy(WebsiteBarRespVO::getPid));
            for (WebsiteBarRespVO websiteBarRespVO : childList) {
                websiteBarRespVO.setBarTypeName(sysGenerator.getUdcName(UdcEnum.WS_WEBSITE_BAR_TYPE_NAV.getCode(), websiteBarRespVO.getBarType()));
                websiteBarRespVO.setTargetTypeName(sysGenerator.getUdcName(UdcEnum.WS_TARGET_TYPE_AD.getCode(), websiteBarRespVO.getTargetType()));
                websiteBarRespVO.setLinkTypeName(sysGenerator.getUdcName(UdcEnum.WS_LINK_TYPE_INTERNAL.getCode(), websiteBarRespVO.getLinkType()));
                websiteBarRespVO.setWindowOpenModeName(sysGenerator.getUdcName(UdcEnum.WS_WINDOW_OPEN_MODE_CURRENT.getCode(), websiteBarRespVO.getWindowOpenMode()));
                websiteBarRespVO.setMallModeName(sysGenerator.getUdcName(UdcEnum.WS_MALL_TYPE_PC_B.getCode(), websiteBarRespVO.getMallMode()));
            }
            // 相同父类的进行分组 设置子类
            for (WebsiteBarRespVO item : parentList) {
                List<WebsiteBarRespVO> childVoList = childMap.get(item.getId());
                if(CollectionUtil.isNotEmpty(childVoList)){
                    childVoList.stream().forEach(child -> child.setPName(item.getBarName()));
                }
                item.setChild(childVoList);
            }
        }
        return PagingVO.<WebsiteBarRespVO>builder()
                .total(parentPage.getTotal())
                .records(parentList).build();
    }

    @Override
    @SysCodeProc
    public List<UpBarRespVO> queryUpBar(String barType) {
        if (!StrUtil.equals(barType, UdcEnum.WS_WEBSITE_BAR_TYPE_NAV.getValueCode())) {
            throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "只有PC导航栏可进行维护");
        }
        List<WebsiteBarDO> doList = websiteBarRepoProc.findByBarTypeAndPidIsNull(barType);
        return WebsiteBarConvert.INSTANCE.dosUpBarVos(doList);
    }
}
