package com.elitescloud.cloudt.system.service.common.constant;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ArrayUtil;
import cn.hutool.core.util.ObjUtil;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.base.SafeEnum;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotNull;
import java.util.Arrays;

/**
 * 归属类型.
 *
 * @author Kaiser（wang shao）
 * 2022/10/13
 */
public class BelongType extends SafeEnum<BelongType> {
    private static final long serialVersionUID = 2642151856062071747L;

    public static final BelongType PLATFORM = new BelongType("platform", "平台");
    public static final BelongType TENANT = new BelongType("tenant", "租户");
    public static final BelongType TENANT_ORG = new BelongType("tenant_org", "租户组织");
    public static final BelongType PERSONAL = new BelongType("personal", "个人");
    /**
     * 自定义
     */
    public static final BelongType CUSTOM = new BelongType("custom", "自定义");

    public BelongType() {
    }

    public BelongType(String value) {
        super(value);
    }

    public BelongType(String value, String description) {
        super(value, description);
    }

    public static BelongType valueOf(String value) {
        return SafeEnum.valueOf(BelongType.class, value);
    }

    /**
     * 获取拥有人
     *
     * @param defaultPersonal 默认是否是当前用户
     * @return 拥有人
     */
    public static Belonger getBelonger(boolean defaultPersonal) {
        GeneralUserDetails currentUser = SecurityContextUtil.currentUser();
        BelongType belongType = null;
        String typeId = null;
        if (currentUser == null) {
            var tenant = SpringContextHolder.getBean(TenantClientProvider.class).getSessionTenant();
            if (tenant != null) {
                belongType = BelongType.TENANT;
                typeId = tenant.getId().toString();
            } else {
                belongType = BelongType.PLATFORM;
                typeId = TenantConstant.DEFAULT_TENANT_ID.toString();
            }
            return new Belonger(belongType, typeId);
        }
        if (currentUser.isOperation()) {
            var tenant = SpringContextHolder.getBean(TenantClientProvider.class).getSessionTenant();
            if (tenant != null) {
                belongType = BelongType.TENANT;
                typeId = tenant.getId().toString();
            } else {
                belongType = BelongType.PLATFORM;
                typeId = TenantConstant.DEFAULT_TENANT_ID.toString();
            }
        } else if (currentUser.isTenantAdmin()) {
            belongType = BelongType.TENANT;
            typeId = currentUser.getTenantId().toString();
        } else if (currentUser.isTenantOrgAdmin()) {
            belongType = BelongType.TENANT_ORG;
            typeId = currentUser.getTenantOrgId().toString();
        } else {
            if (defaultPersonal) {
                belongType = BelongType.PERSONAL;
                typeId = currentUser.getUserId().toString();
            } else {
                belongType = BelongType.CUSTOM;
                typeId = "";
            }
        }

        return new Belonger(belongType, typeId);
    }

    public static String getBelongerId(@NotNull BelongType belongType, String... belongId) {
        Assert.notNull(belongType, "归属者类型为空");
        GeneralUserDetails currentUser = SecurityContextUtil.currentUserIfUnauthorizedThrow();
        if (belongType == PLATFORM) {
            return TenantConstant.DEFAULT_TENANT_ID.toString();
        }
        if (belongType == TENANT) {
            return ObjUtil.defaultIfNull(currentUser.getTenantId(), TenantConstant.DEFAULT_TENANT_ID).toString();
        }
        if (belongType == TENANT_ORG) {
            return ObjUtil.defaultIfNull(currentUser.getTenantOrgId(), "").toString();
        }
        if (belongType == PERSONAL) {
            return ObjUtil.defaultIfNull(currentUser.getUserId(), "").toString();
        }

        if (ArrayUtil.isEmpty(belongId)) {
            return "";
        }
        String[] belongerIds = Arrays.stream(belongId).filter(StringUtils::hasText).toArray(String[]::new);

        return belongerIds.length == 0 ? "" : String.join(":", belongerIds);
    }

    /**
     * 获取个人拥有者
     *
     * @return
     */
    public static Belonger getBelonger() {
        return getBelonger(false);
    }

    /**
     * 获取个人拥有者
     *
     * @return
     */
    public static Belonger getBelongerPersonal() {
        GeneralUserDetails currentUser = SecurityContextUtil.currentUserIfUnauthorizedThrow();
        return new Belonger(BelongType.PERSONAL, currentUser.getUserId().toString());
    }

    public static class Belonger {
        private final BelongType belongType;
        private final String belongId;

        public Belonger(BelongType belongType, String belongId) {
            this.belongType = belongType;
            this.belongId = ObjUtil.defaultIfNull(belongId, "");
        }

        public BelongType getBelongType() {
            return belongType;
        }

        public String getBelongId() {
            return belongId;
        }
    }
}
