package com.elitescloud.cloudt.system.service.repo;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.system.service.model.entity.*;
import com.elitescloud.cloudt.system.service.model.vo.SysUserTenantVO;
import com.querydsl.core.Tuple;
import com.querydsl.core.types.Projections;
import org.springframework.stereotype.Repository;

import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/3/23
 */
@Repository
public class SysTenantUserRepoProc extends BaseRepoProc<SysTenantUserDO> {

    private static final QSysTenantUserDO QDO = QSysTenantUserDO.sysTenantUserDO;
    private static final QSysTenantDO QDO_TENANT = QSysTenantDO.sysTenantDO;
    private static final QSysUserDO QDO_USER = QSysUserDO.sysUserDO;

    public SysTenantUserRepoProc() {
        super(QDO);
    }

    /**
     * 根据租住删除
     *
     * @param tenantId 租户ID
     */
    public void deleteByTenant(long tenantId) {
        super.deleteByValue(QDO.sysTenantId, tenantId);
    }

    /**
     * 解绑租户与用户关系
     *
     * @param tenantId
     * @param userId
     */
    public void unbindTenant(long tenantId, long userId) {
        jpaQueryFactory.delete(QDO)
                .where(QDO.sysUserId.eq(userId).and(QDO.sysTenantId.eq(tenantId)))
                .execute();
    }

    /**
     * 更新启用状态
     *
     * @param tenantId
     * @param userId
     * @param enabled
     */
    public void updateEnabled(Long tenantId, Long userId, Boolean enabled) {
        jpaQueryFactory.update(QDO)
                .set(QDO.enabled, enabled)
                .where(QDO.sysUserId.eq(userId).and(QDO.sysTenantId.eq(tenantId)))
                .execute();
    }

    /**
     * 获取用户租户关联
     *
     * @param tenantId 租户ID
     * @param userId   用户ID
     * @return 关联
     */
    public SysTenantUserDO get(Long tenantId, Long userId) {
        return jpaQueryFactory.select(QDO)
                .from(QDO)
                .where(QDO.sysUserId.eq(userId).and(QDO.sysTenantId.eq(tenantId)))
                .limit(1)
                .fetchOne();
    }

    /**
     * 获取用户的租户ID
     *
     * @param userId
     * @return
     */
    public List<Long> getTenantIdOfUser(Long userId) {
        return jpaQueryFactory.select(QDO.sysTenantId)
                .from(QDO)
                .where(QDO.sysUserId.eq(userId))
                .fetch();
    }

    /**
     * 查询用户绑定的租户详细信息
     *
     * @param userId 用户ID
     * @return
     */
    public List<SysTenantDO> queryUserTenantDetail(Long userId) {
        return jpaQueryFactory.select(QDO_TENANT)
                .from(QDO).leftJoin(QDO_TENANT).on(QDO_TENANT.id.eq(QDO.sysTenantId))
                .where(QDO.sysUserId.eq(userId).and(QDO_TENANT.deleteFlag.eq(0)).and(QDO.enabled.eq(true)))
                .orderBy(QDO.lastLoginTime.desc())
                .fetch();

    }

    /**
     * 获取用户绑定的租户信息
     *
     * @param userId 用户ID
     * @return
     */
    public List<SysUserTenantVO> queryUserTenant(Long userId) {
        return jpaQueryFactory.select(Projections.bean(SysUserTenantVO.class,
                                QDO.sysUserId, QDO.sysTenantId, QDO.bindTime,
                                QDO_TENANT.tenantCode, QDO_TENANT.tenantName, QDO_TENANT.tenantDomain, QDO_TENANT.customDomain
                        )
                ).from(QDO)
                .leftJoin(QDO_TENANT).on(QDO_TENANT.id.eq(QDO.sysTenantId))
                .where(QDO.sysUserId.eq(userId).and(QDO_TENANT.enabled.eq(true)))
                .fetch();
    }

    /**
     * 过滤存在的账户ID
     *
     * @param userIds
     * @return
     */
    public List<Long> filterExistsUserIds(Collection<Long> userIds, Long tenantId) {
        var predicate = PredicateBuilder.builder()
                .andIn(QDO.sysUserId, userIds)
                .andEq(QDO.sysTenantId, tenantId)
                .build();
        return super.getValueList(QDO.sysUserId, predicate);
    }

    public List<IdCodeNameParam> queryUsers(Collection<Long> userIds, Long tenantId) {
        var predicate = PredicateBuilder.builder()
                .andIn(QDO.sysUserId, userIds)
                .andEq(QDO.sysTenantId, tenantId)
                .build();
        return jpaQueryFactory.select(QDO_USER.id, QDO_USER.username, QDO_USER.firstName, QDO_USER.lastName)
                .from(QDO_USER)
                .leftJoin(QDO).on(QDO.sysUserId.eq(QDO_USER.id))
                .where(predicate)
                .fetch()
                .stream()
                .map(t -> {
                    IdCodeNameParam param = new IdCodeNameParam();
                    param.setId(t.get(QDO_USER.id));
                    param.setCode(t.get(QDO_USER.username));
                    param.setName(obtainUserFullName(t, QDO_USER));
                    return param;
                }).collect(Collectors.toList());
    }

    private String obtainUserFullName(Tuple tuple, QSysUserDO user) {
        String firstName = tuple.get(user.firstName);
        String lastName = tuple.get(user.lastName);
        return CharSequenceUtil.isNotBlank(lastName) ? lastName + CharSequenceUtil.blankToDefault(firstName, "") : firstName;
    }
}
