package com.elitescloud.cloudt.system.service.repo;

import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.system.service.model.entity.QSysPlatformAppDO;
import com.elitescloud.cloudt.system.service.model.entity.QSysTenantAppDO;
import com.elitescloud.cloudt.system.service.model.entity.SysTenantAppDO;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.QBean;
import com.querydsl.jpa.JPAExpressions;
import org.springframework.stereotype.Repository;
import org.springframework.util.CollectionUtils;

import javax.validation.constraints.NotBlank;
import java.util.*;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/9/20
 */
@Repository
public class SysTenantAppRepoProc extends BaseRepoProc<SysTenantAppDO> {
    private static final QSysTenantAppDO QDO = QSysTenantAppDO.sysTenantAppDO;
    private static final QSysPlatformAppDO QDO_APP = QSysPlatformAppDO.sysPlatformAppDO;

    public SysTenantAppRepoProc() {
        super(QDO);
    }


    /**
     * 根据应用编码删除
     *
     * @param appCode 应用编码
     */
    public void deleteByApp(@NotBlank String appCode) {
        super.deleteByValue(QDO.appCode, appCode);
    }


    /**
     * 根据租户删除
     *
     * @param tenantId 应用编码
     */
    public void deleteByTenant(long tenantId) {
        super.deleteByValue(QDO.sysTenantId, tenantId);
    }

    public void deleteByTenantAndAppCode(Long sysTenantId, Set<String> appCodes) {
        jpaQueryFactory.delete(QDO)
                .where(QDO.sysTenantId.eq(sysTenantId).and(QDO.appCode.in(appCodes)))
                .execute();
    }

    public Set<String> getAppCode(Long sysTenantId) {
        return new HashSet<>(jpaQueryFactory.select(QDO.appCode)
                .from(QDO)
                .where(QDO.sysTenantId.eq(sysTenantId))
                .fetch());
    }

    public Map<Long, Set<String>> getAppCodes(Collection<Long> tenantIds) {
        return jpaQueryFactory.select(QDO.sysTenantId, QDO.appCode)
                .from(QDO)
                .where(CollectionUtils.isEmpty(tenantIds) ? null : QDO.sysTenantId.in(tenantIds))
                .fetch()
                .stream()
                .collect(Collectors.groupingBy(t -> t.get(QDO.sysTenantId),
                        Collectors.collectingAndThen(Collectors.toList(),
                                t -> t.stream().map(tt -> tt.get(QDO.appCode)).collect(Collectors.toSet()))));
    }

    public Map<Long, Set<String>> getAppCodeAll() {
        return jpaQueryFactory.select(QDO.sysTenantId, QDO.appCode)
                .from(QDO)
                .fetch()
                .stream()
                .collect(Collectors.groupingBy(t -> t.get(QDO.sysTenantId),
                        Collectors.collectingAndThen(Collectors.toList(),
                                t -> t.stream().map(tt -> tt.get(QDO.appCode)).collect(Collectors.toSet()))));
    }

    /**
     * 获取租户的应用信息
     *
     * @param sysTenantId
     * @return
     */
    public List<CodeNameParam> getAppOfTenant(long sysTenantId) {
        return super.jpaQueryFactory.select(this.qBeanCodeName())
                .from(QDO_APP)
                .where(QDO_APP.appCode.in(
                        JPAExpressions.select(QDO.appCode).from(QDO).where(QDO.sysTenantId.eq(sysTenantId))
                ).and(QDO_APP.appState.eq(true)))
                .orderBy(QDO_APP.appOrder.desc())
                .fetch();
    }

    private QBean<CodeNameParam> qBeanCodeName() {
        return Projections.bean(CodeNameParam.class, QDO_APP.appCode, QDO_APP.appName);
    }
}
