package com.elitescloud.cloudt.system.service.manager;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.boot.provider.TenantDataIsolateProvider;
import com.elitescloud.boot.redis.util.RedisUtils;
import com.elitescloud.cloudt.context.util.CollectionUtil;
import com.elitescloud.cloudt.context.util.TreeDataUtil;
import com.elitescloud.cloudt.system.service.convert.AreaConvert;
import com.elitescloud.cloudt.system.service.model.entity.SysPlatformAreaDO;
import com.elitescloud.cloudt.system.cacheable.SysCacheAreaRpcService;
import com.elitescloud.cloudt.system.dto.resp.SysAreaRespDTO;
import com.elitescloud.cloudt.system.service.model.vo.CommonAreaTreeRespVO;
import com.elitescloud.cloudt.system.service.model.bo.AreaBO;
import com.elitescloud.cloudt.system.service.repo.AreaRepoProc;
import com.elitescloud.cloudt.system.service.repo.TenantAreaRepoProc;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.function.Supplier;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2023/10/18
 */
@Component
public class AreaManager {

    @Autowired
    private AreaRepoProc repoProc;
    @Autowired
    private TenantAreaRepoProc tenantAreaRepoProc;

    @Autowired
    private TenantDataIsolateProvider tenantDataIsolateProvider;
    @Autowired
    private TenantClientProvider tenantClientProvider;
    @Autowired
    private RedisUtils redisUtils;

    @Autowired
    private SysCacheAreaRpcService cacheAreaRpcService;

    private static final String CACHE_TREE_COMMON = "cloudt:sysAreaTreeMap";
    private static final String CACHE_ITEM_ALL_DTO = "allDto";
    private static final String CACHE_ITEM_ALL_VO = "allVo";
    private static final String CACHE_ITEM_CN_VO = "CnVo";

    /**
     * 获取行政区域信息
     *
     * @param provinceCode
     * @param cityCode
     * @param countyCode
     * @return
     */
    public AreaBO getAreaBO(String provinceCode, String cityCode, String countyCode) {
        List<String> codes = new ArrayList<>(8);
        if (CharSequenceUtil.isNotBlank(provinceCode)) {
            codes.add(provinceCode);
        }
        if (CharSequenceUtil.isNotBlank(cityCode)) {
            codes.add(cityCode);
        }
        if (CharSequenceUtil.isNotBlank(countyCode)) {
            codes.add(countyCode);
        }
        if (codes.isEmpty()) {
            return null;
        }

        var codeNameMap = repoProc.getCodeAndName(codes);

        AreaBO areaBO = new AreaBO();
        areaBO.setProvinceCode(provinceCode);
        if (CharSequenceUtil.isNotBlank(provinceCode)) {
            areaBO.setProvinceName(codeNameMap.get(provinceCode));
        }
        areaBO.setCityCode(cityCode);
        if (CharSequenceUtil.isNotBlank(cityCode)) {
            areaBO.setCityName(codeNameMap.get(cityCode));
        }
        areaBO.setCountyCode(countyCode);
        if (CharSequenceUtil.isNotBlank(countyCode)) {
            areaBO.setCountyName(codeNameMap.get(countyCode));
        }

        return areaBO;
    }

    /**
     * 清空缓存
     */
    public void clearCache() {
        redisUtils.hdel(CACHE_TREE_COMMON);
        cacheAreaRpcService.clearCache();
    }

    /**
     * 行政区域树
     *
     * @param tree
     * @return
     */
    public List<SysAreaRespDTO> allTree(boolean tree) {
        // 先从缓存中获取
        List<SysAreaRespDTO> dataTree = (List<SysAreaRespDTO>) redisUtils.hget(CACHE_TREE_COMMON, CACHE_ITEM_ALL_DTO);
        if (dataTree != null) {
            if (!tree) {
                dataTree = CollectionUtil.expandTree(dataTree, SysAreaRespDTO::getChildren);
            }
            return dataTree;
        }

        // 从数据库查询，并缓存起来
        dataTree = this.queryAreaTree();
        redisUtils.hset(CACHE_TREE_COMMON, CACHE_ITEM_ALL_DTO, dataTree);

        if (tree) {
            return dataTree;
        }
        return CollectionUtil.expandTree(dataTree, SysAreaRespDTO::getChildren);
    }

    /**
     * 行政区域树
     *
     * @param pid
     * @param pcode
     * @param tree
     * @return
     */
    public List<CommonAreaTreeRespVO> listTree(Long pid, String pcode, boolean tree) {
        List<CommonAreaTreeRespVO> respVoList = null;
        if (pid == null && CharSequenceUtil.isBlank(pcode)) {
            // 所有的
            respVoList = (List<CommonAreaTreeRespVO>) redisUtils.hget(CACHE_TREE_COMMON, CACHE_ITEM_ALL_VO);
            if (respVoList != null) {
                return tree ? respVoList : CollectionUtil.expandTree(respVoList, CommonAreaTreeRespVO::getChildren);
            }
            respVoList = queryAreaTree(null, null);
            redisUtils.hset(CACHE_TREE_COMMON, CACHE_ITEM_ALL_VO, respVoList);
            return tree ? respVoList : CollectionUtil.expandTree(respVoList, CommonAreaTreeRespVO::getChildren);
        }

        // 中国的
        boolean isCn = SysPlatformAreaDO.CODE_CN.equals(pcode);
        if (!isCn && pid != null) {
            var tempCode = repoProc.getCode(pid);
            isCn = SysPlatformAreaDO.CODE_CN.equals(tempCode);
        }
        if (isCn) {
            respVoList = (List<CommonAreaTreeRespVO>) redisUtils.hget(CACHE_TREE_COMMON, CACHE_ITEM_CN_VO);
            if (respVoList != null) {
                return tree ? respVoList : CollectionUtil.expandTree(respVoList, CommonAreaTreeRespVO::getChildren);
            }
            respVoList = queryAreaTree(pid, pcode);
            // 去掉中国
            if (respVoList.size() == 1 && SysPlatformAreaDO.CODE_CN.equals(respVoList.get(0).getCode())) {
                respVoList = respVoList.get(0).getChildren();
            }
            redisUtils.hset(CACHE_TREE_COMMON, CACHE_ITEM_CN_VO, respVoList);
            return tree ? respVoList : CollectionUtil.expandTree(respVoList, CommonAreaTreeRespVO::getChildren);
        }

        // 默认直接查库
        respVoList = queryAreaTree(pid, pcode);
        return tree ? respVoList : CollectionUtil.expandTree(respVoList, CommonAreaTreeRespVO::getChildren);
    }

    public <T> T areaTemplate(Supplier<T> supplier) {
        var enabled = this.enabledPlatform();
        return enabled ? tenantDataIsolateProvider.byDefaultDirectly(supplier) : supplier.get();
    }

    @SuppressWarnings("unchecked")
    private List<SysAreaRespDTO> queryAreaTree() {
        var dtoList = areaTemplate(() -> repoProc.all())
                .stream()
                .filter(t -> Boolean.TRUE.equals(t.getEnabled()))
                .map(t -> {
                    var dto = AreaConvert.INSTANCE.do2DTO(t);
                    if (dto.getSortNo() == null) {
                        dto.setSortNo(0);
                    }
                    return dto;
                })
                .collect(Collectors.toList());
        if (dtoList.isEmpty()) {
            return Collections.emptyList();
        }

        // 转树形
        TreeDataUtil<SysAreaRespDTO> treeDataUtil = new TreeDataUtil<>(dtoList, SysAreaRespDTO::getAreaCode, SysAreaRespDTO::getParentAreaCode, SysAreaRespDTO::setChildren, Comparator.comparingInt(SysAreaRespDTO::getSortNo));
        return (List<SysAreaRespDTO>) treeDataUtil.getRoots();
    }

    private List<CommonAreaTreeRespVO> queryAreaTree(Long pid, String pcode) {
        var respVoList = repoProc.queryList(pcode, pid);
        if (respVoList.isEmpty()) {
            return respVoList;
        }

        // 转树形
        TreeDataUtil<CommonAreaTreeRespVO> treeDataUtil = new TreeDataUtil<>(respVoList, CommonAreaTreeRespVO::getCode, CommonAreaTreeRespVO::getParentCode, CommonAreaTreeRespVO::setChildren, Comparator.comparingInt(CommonAreaTreeRespVO::getSortNo));
        return (List<CommonAreaTreeRespVO>) treeDataUtil.getRoots();
    }

    private boolean enabledPlatform() {
        var tenant = tenantClientProvider.getCurrentTenant();
        var tenantId = tenant == null ? TenantConstant.DEFAULT_TENANT_ID : tenant.getId();

        var enabled = tenantAreaRepoProc.getEnabled(tenantId);
        return enabled == null || enabled;
    }
}
