package com.elitescloud.cloudt.system.service.repo;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.jpa.common.BaseTreeRepoProc;
import com.elitescloud.boot.model.entity.BaseTreeModel;
import com.elitescloud.boot.util.ObjUtil;
import com.elitescloud.cloudt.system.dto.req.SysAreaQueryDTO;
import com.elitescloud.cloudt.system.service.model.entity.QSysPlatformAreaDO;
import com.elitescloud.cloudt.system.service.model.entity.SysPlatformAreaDO;
import com.elitescloud.cloudt.system.service.model.vo.CommonAreaQueryVO;
import com.elitescloud.cloudt.system.service.model.vo.CommonAreaTreeRespVO;
import com.querydsl.core.types.Projections;
import org.springframework.stereotype.Repository;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.util.*;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2023/1/31
 */
@Repository
public class AreaRepoProc extends BaseTreeRepoProc<SysPlatformAreaDO> {
    private static final QSysPlatformAreaDO QDO = QSysPlatformAreaDO.sysPlatformAreaDO;

    public AreaRepoProc() {
        super(QDO);
    }

    /**
     * 判断编码是否存在
     *
     * @param code
     * @return
     */
    public boolean existsCode(@NotBlank String code) {
        return super.exists(QDO.areaCode, code);
    }

    /**
     * 获取编码
     *
     * @param id
     * @return
     */
    public String getCode(long id) {
        return super.getValue(QDO.areaCode, id);
    }

    /**
     * 根据行政区域编码查询行政区域
     *
     * @param codes 区域编码
     * @return 区域列表
     */
    public List<SysPlatformAreaDO> listByCode(@NotEmpty Collection<String> codes) {
        return super.getListByValue(QDO.areaCode, codes);
    }

    /**
     * 根据行政区域编码或名称查询行政区域
     *
     * @param codeOrNames 区域编码或名称
     * @return 区域列表
     */
    public List<SysPlatformAreaDO> listByCodeOrName(@NotEmpty Collection<String> codeOrNames) {
        return super.getList(QDO.areaCode.in(codeOrNames).or(QDO.areaName.in(codeOrNames)));
    }

    /**
     * 根据行政区域编码或名称查询行政区域
     *
     * @param codeOrNames 区域编码或名称
     * @return 区域列表
     */
    public List<SysPlatformAreaDO> listByCodeOrNameFuzzily(@NotEmpty Collection<String> codeOrNames) {
        var predicate = PredicateBuilder.builder()
                .andRightLike(QDO.areaCode, codeOrNames)
                .andRightLike(QDO.areaName, codeOrNames)
                .buildOr();
        return super.getList(predicate);
    }

    /**
     * 根据上级行政区域编码查询行政区域
     *
     * @param parentAreaCode 区域编码
     * @return 区域列表
     */
    public List<SysPlatformAreaDO> listByParentAreaCode(@NotBlank String parentAreaCode) {
        return super.getListByValue(QDO.parentAreaCode, parentAreaCode);
    }

    /**
     * 查询行政区域列表
     *
     * @param queryDTO 查询参数
     * @return 区域列表
     */
    public List<SysPlatformAreaDO> queryList(@NotNull SysAreaQueryDTO queryDTO) {
        Set<String> codeOrNames = CollUtil.isEmpty(queryDTO.getCodeOrNames()) ? Collections.emptySet() :
                queryDTO.getCodeOrNames().stream().filter(StringUtils::hasText).collect(Collectors.toSet());

        var predicate = PredicateBuilder.builder()
                .andEq(QDO.areaCode, queryDTO.getAreaCode())
                .andIn(QDO.areaCode, queryDTO.getAreaCodes())
                .andEq(QDO.parentAreaCode, queryDTO.getParentAreaCode())
                .andEq(QDO.areaType, queryDTO.getAreaType())
                .andEq(QDO.enabled, queryDTO.getEnabled())
                .andEq(QDO.areaName, queryDTO.getAreaName())
                .andIn(QDO.areaName, queryDTO.getAreaNames())
                .andRightLike(QDO.areaName, queryDTO.getAreaNameRightLike())
                .andLike(QDO.pinyin, queryDTO.getPinyin())
                .and(StringUtils.hasText(queryDTO.getCodeOrName()), () -> QDO.areaCode.eq(queryDTO.getCodeOrName()).or(QDO.areaName.eq(queryDTO.getCodeOrName())))
                .and(CollUtil.isNotEmpty(codeOrNames), () -> QDO.areaCode.in(codeOrNames).or(QDO.areaName.in(codeOrNames)))
                .build();

        return super.getList(predicate);
    }

    /**
     * 查询区域列表
     *
     * @param queryVO
     * @return
     */
    public List<CommonAreaTreeRespVO> queryList(@NotNull CommonAreaQueryVO queryVO) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.pId, queryVO.getPid())
                .andEq(QDO.parentAreaCode, queryVO.getPcode())
                .andEq(QDO.areaType, queryVO.getAreaType())
                .andLike(QDO.areaName, queryVO.getName())
                .andEq(true, QDO.enabled, true)
                .build();

        var pathChildrenNum = QDO.rgt.subtract(QDO.lft.add(1)).divide(2).as("childrenNum");
        return super.jpaQueryFactory.select(Projections.bean(CommonAreaTreeRespVO.class, QDO.id, QDO.areaCode.as("code"),
                        QDO.areaName.as("name"), QDO.sortNo, QDO.pId.as("parentId"), QDO.parentAreaCode.as("parentCode"),
                        pathChildrenNum))
                .from(QDO)
                .where(predicate)
                .orderBy(QDO.sortNo.asc())
                .fetch();
    }

    /**
     * 查询区域列表
     *
     * @param pcode
     * @param pid
     * @return
     */
    public List<CommonAreaTreeRespVO> queryList(String pcode, Long pid) {
        if (pid == null && CharSequenceUtil.isNotBlank(pcode)) {
            pid = getIdByValue(QDO.areaCode, pcode);
        }

        return getTree(pid, null, t -> Boolean.TRUE.equals(t.getEnabled()), t -> {
            CommonAreaTreeRespVO respVO = new CommonAreaTreeRespVO();
            respVO.setId(t.getId());
            respVO.setCode(t.getAreaCode());
            respVO.setName(CharSequenceUtil.blankToDefault(t.getShortName(), t.getAreaName()));
            respVO.setSortNo(ObjUtil.defaultIfNull(t.getSortNo(), 0));
            respVO.setParentId(t.getPId());
            respVO.setParentCode(t.getParentAreaCode());

            return respVO;
        });
    }

    /**
     * 获取编码与ID
     *
     * @return
     */
    public Map<String, String> getCodeAndName(@NotEmpty Collection<String> areaCodes) {
        return super.jpaQueryFactory.select(QDO.areaCode, QDO.areaName)
                .from(QDO)
                .where(QDO.areaCode.in(areaCodes))
                .fetch()
                .stream()
                .collect(Collectors.toMap(t -> t.get(QDO.areaCode), t -> t.get(QDO.areaName), (t1, t2) -> t1));
    }
}
