package com.elitescloud.cloudt.tenant.controller;

import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.tenant.model.vo.SysTenantVO;
import com.elitescloud.cloudt.tenant.model.vo.params.SysTenantCreateParam;
import com.elitescloud.cloudt.tenant.model.vo.params.SysTenantQueryParam;
import com.elitescloud.cloudt.tenant.model.vo.params.SysTenantUpdateParam;
import com.elitescloud.cloudt.tenant.service.SysTenantMngService;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;

/**
 * 租户管理
 *
 * @author roman.zhang
 * @since 2022-02-16 15:15:58
 */
@ResponseBody
@RequestMapping("/sys/sysTenant")
@Api(value = "租户管理", tags = {"租户管理"})
public class SysTenantController {

    @Autowired
    private SysTenantMngService sysTenantService;

    /**
     * 创建租户
     *
     * @param param 租户数据
     * @return 新增数据情况
     */
    @PostMapping("/createOne")
    @ApiOperation(value = "创建租户")
    @ApiOperationSupport(order = 1)
    public ApiResult<Long> createOne(@Valid @RequestBody SysTenantCreateParam param) {
        return sysTenantService.add(param);
    }

    /**
     * 更新租户
     *
     * @param param 修改对象
     * @return 修改数据详情
     */
    @PutMapping("/update")
    @ApiOperation("更新租户")
    @ApiOperationSupport(order = 2)
    public ApiResult<Long> update(@Valid @RequestBody SysTenantUpdateParam param) {
        return sysTenantService.update(param);
    }

    /**
     * 通过ID查询租户详细信息
     *
     * @param id 主键
     * @return 租户详细信息
     */
    @GetMapping("/findIdOne/{id}")
    @ApiOperation("通过ID查询租户详细信息")
    @ApiOperationSupport(order = 3)
    ApiResult<SysTenantVO> findIdOne(@PathVariable Long id) {
        return sysTenantService.getDetail(id);
    }


    /**
     * 分页查询租户
     *
     * @param param 分类查询、分页和排序对象
     * @return 多条数据
     */
    @PostMapping("/search")
    @ApiOperation("分页查询租户")
    @ApiOperationSupport(order = 4)
    public ApiResult<PagingVO<SysTenantVO>> search(@RequestBody SysTenantQueryParam param) {
        return sysTenantService.search(param);
    }

    /**
     * 启用租户
     *
     * @param id 租户ID
     * @return 启用结果
     */
    @ApiOperation("启用租户")
    @ApiOperationSupport(order = 11)
    @PatchMapping(value = "/{id}/enabled")
    public ApiResult<Long> enabled(@PathVariable Long id) {
        return sysTenantService.updateEnabled(id, true);
    }

    /**
     * 禁用租户
     *
     * @param id 租户ID
     * @return 启用结果
     */
    @ApiOperation("禁用租户")
    @ApiOperationSupport(order = 12)
    @PatchMapping(value = "/{id}/disabled")
    public ApiResult<Long> disabled(@PathVariable Long id) {
        return sysTenantService.updateEnabled(id, false);
    }

    /**
     * 删除租户
     *
     * @param id 主键
     * @return 删除情况
     */
    @DeleteMapping("/deleteOne/{id}")
    @ApiOperation(value = "删除租户")
    @ApiImplicitParam(name = "id", value = "租户ID", paramType = "path", required = true)
    public ApiResult<Long> deleteOne(@PathVariable("id") Long id) {
        return sysTenantService.deleteById(id);
    }

    /**
     * 逻辑删除
     *
     * @param id 更新数据
     */
    @PutMapping("/updateDeleteFlag/{id}")
    @ApiOperation(value = "逻辑删除", hidden = true)
    @ApiImplicitParam(name = "id", value = "租户ID", paramType = "path", required = true)
    public ApiResult<Object> updateDeleteFlag(@PathVariable("id") Long id) {
        sysTenantService.updateDeleteFlag(id);
        return ApiResult.ok();
    }

}


