package com.elitescloud.cloudt.tenant.service.impl;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.RandomUtil;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.constant.Gender;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.tenant.client.TenantClientProperties;
import com.elitescloud.boot.wrapper.RedisWrapper;
import com.elitescloud.cloudt.Application;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.constant.TenantIsolateStrategy;
import com.elitescloud.cloudt.common.constant.Terminal;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.constant.TenantType;
import com.elitescloud.cloudt.system.constant.UserSourceType;
import com.elitescloud.cloudt.system.constant.UserType;
import com.elitescloud.cloudt.system.dto.SysUserTypeDTO;
import com.elitescloud.cloudt.system.dto.req.UserCreateDTO;
import com.elitescloud.cloudt.system.service.IUserService;
import com.elitescloud.cloudt.system.service.callback.TenantChangedCallback;
import com.elitescloud.cloudt.system.service.model.entity.SysTenantAppDO;
import com.elitescloud.cloudt.system.service.model.entity.SysTenantDO;
import com.elitescloud.cloudt.system.service.model.entity.SysTenantUserDO;
import com.elitescloud.cloudt.system.service.model.vo.SysUserTenantVO;
import com.elitescloud.cloudt.system.service.repo.SysTenantAppRepo;
import com.elitescloud.cloudt.system.service.repo.SysTenantAppRepoProc;
import com.elitescloud.cloudt.system.service.repo.SysTenantUserRepo;
import com.elitescloud.cloudt.system.service.repo.SysTenantUserRepoProc;
import com.elitescloud.cloudt.system.service.config.TenantProperties;
import com.elitescloud.cloudt.tenant.convert.SysTenantConvert;
import com.elitescloud.cloudt.tenant.model.vo.SysTenantVO;
import com.elitescloud.cloudt.tenant.model.vo.params.SysTenantCreateParam;
import com.elitescloud.cloudt.tenant.model.vo.params.SysTenantQueryParam;
import com.elitescloud.cloudt.tenant.model.vo.params.SysTenantUpdateParam;
import com.elitescloud.cloudt.tenant.service.SysTenantMngService;
import com.elitescloud.cloudt.tenant.service.manager.SysTenantManager;
import com.elitescloud.cloudt.tenant.service.repo.SysTenantRepo;
import com.elitescloud.cloudt.tenant.service.repo.SysTenantRepoProc;
import lombok.RequiredArgsConstructor;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.time.LocalDateTime;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * <p>
 * 功能说明
 * </p>
 *
 * @author roman.zhang
 * @since 2022-02-16 15:16:01
 */
@Log4j2
@RequiredArgsConstructor
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
@TenantOrgTransaction(useTenantOrg = false)
public class SysTenantMngServiceImpl extends BaseServiceImpl implements SysTenantMngService {
    private static final SysTenantConvert CONVERT = SysTenantConvert.INSTANCE;
    @Autowired
    private SysTenantRepo sysTenantRepo;
    @Autowired
    private SysTenantRepoProc sysTenantRepoProc;
    @Autowired
    private SysTenantUserRepo sysTenantUserRepo;
    @Autowired
    private SysTenantUserRepoProc sysTenantUserRepoProc;
    @Autowired
    private SysTenantAppRepo tenantAppRepo;
    @Autowired
    private SysTenantAppRepoProc tenantAppRepoProc;
    @Autowired
    private TenantProperties tenantProperties;
    @Autowired
    private TenantClientProperties tenantClientProperties;
    @Autowired
    private SysTenantManager tenantManager;
    @Autowired
    private IUserService userService;
    @Autowired
    private RedisWrapper redisWrapper;
    @Autowired
    private ObjectProvider<TenantChangedCallback> tenantChangedCallbacks;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> add(SysTenantCreateParam createParam) {
        try {
            checkForSave(createParam);
        } catch (IllegalArgumentException e) {
            return ApiResult.fail("创建租户失败，" + e.getMessage());
        }

        // 保存租户管理员信息
        long adminUserId = createTenantAdmin(createParam);

        // 转成租户信息
        var tenantDO = CONVERT.saveParam2Do(createParam);
        tenantDO.setSysUserId(adminUserId);
        tenantDO.setTenantType(TenantType.COMMON.getValue());
        // 是否自动创建db
        boolean dbInitialized = Boolean.TRUE.equals(tenantProperties.isAutoSyncDb())
                && (CharSequenceUtil.equals(tenantDO.getTenantIsolation(), TenantIsolateStrategy.SCHEMA.name())
                || CharSequenceUtil.equals(tenantDO.getTenantIsolation(), TenantIsolateStrategy.DATABASE.name()));
        if (TenantIsolateStrategy.FIELD.name().equals(tenantDO.getTenantIsolation())) {
            dbInitialized = true;
        }
        tenantDO.setDbInitialized(dbInitialized);
        tenantDO.setCreateTime(LocalDateTime.now());

        // 保存租户信息
        sysTenantRepo.save(tenantDO);

        if (Boolean.TRUE.equals(tenantProperties.isAutoSyncDb())) {
            // 同步创建租户客户端db
            tenantManager.syncClientDb(tenantDO);
        }

        // 绑定租户与管理员的关系
        bindTenantUser(adminUserId, tenantDO.getId());

        // 清理租户缓存
        clearCache();

        // 回调
        for (TenantChangedCallback callback : tenantChangedCallbacks) {
            callback.onSave(true, tenantDO.getId());
        }
        return ApiResult.ok(tenantDO.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> update(SysTenantUpdateParam updateParam) {
        var tenantDO = sysTenantRepo.findById(updateParam.getId()).orElse(null);
        try {
            checkForUpdate(updateParam, tenantDO);
        } catch (IllegalArgumentException e) {
            return ApiResult.fail("修改租户失败，" + e.getMessage());
        }

        // 更新管理员信息
        var adminId = updateTenantAdmin(updateParam, tenantDO.getSysUserId());
        tenantDO.setSysUserId(adminId);

        CONVERT.convertUpdateParam(updateParam, tenantDO);
        tenantDO.setModifyTime(LocalDateTime.now());

        sysTenantRepo.save(tenantDO);

        // 绑定租户与管理员的关系
        bindTenantUser(adminId, tenantDO.getId());

        // 清理租户缓存
        clearCache();

        // 回调
        for (TenantChangedCallback callback : tenantChangedCallbacks) {
            callback.onSave(false, tenantDO.getId());
        }
        return ApiResult.ok(tenantDO.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateEnabled(Long id, Boolean enabled) {
        GeneralUserDetails user = SecurityContextUtil.currentUserIfUnauthorizedThrow();
        if (!user.isOperation()) {
            return ApiResult.fail("无权限操作");
        }
        sysTenantRepoProc.updateEnabled(id, Boolean.TRUE.equals(enabled));

        // 清理租户缓存
        clearCache();

        // 回调
        for (TenantChangedCallback callback : tenantChangedCallbacks) {
            callback.onEnabled(id, Boolean.TRUE.equals(enabled));
        }
        return ApiResult.ok(id);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Long> deleteById(Long id) {
        GeneralUserDetails user = SecurityContextUtil.currentUserIfUnauthorizedThrow();
        if (!user.isOperation()) {
            return ApiResult.fail("无权限操作");
        }

        Boolean dbInitialized = sysTenantRepoProc.isDbInitialized(id);
        if (dbInitialized != null && dbInitialized) {
            return ApiResult.fail("请先清理表结构");
        }

        this.sysTenantRepoProc.delete(id);
        // 清理租户缓存
        clearCache();

        // 回调
        for (TenantChangedCallback callback : tenantChangedCallbacks) {
            callback.onDelete(id, true);
        }
        return ApiResult.ok(id);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Long> updateDeleteFlag(Long id) {
        GeneralUserDetails user = SecurityContextUtil.currentUserIfUnauthorizedThrow();
        if (!user.isOperation()) {
            return ApiResult.fail("无权限操作");
        }
        sysTenantRepoProc.updateDeleteFlag(id);

        // 清理租户缓存
        clearCache();

        // 回调
        for (TenantChangedCallback callback : tenantChangedCallbacks) {
            callback.onDelete(id, false);
        }
        return ApiResult.ok(id);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Long> updateUserBindTenant(Long userId, Long tenantId) {
        bindTenantUser(userId, tenantId);

        return ApiResult.ok(userId);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ApiResult<Long> updateUserUnbindTenant(Long userId, Long tenantId) {
        sysTenantUserRepoProc.unbindTenant(tenantId, userId);
        return ApiResult.ok(userId);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> saveApps(Long tenantId, Set<String> appCode) {
        if (tenantId == null) {
            return ApiResult.fail("租户ID为空");
        }

        // 目前已有的app
        Set<String> existsCodes = tenantAppRepoProc.getAppCode(tenantId);

        // 需要新增的app
        LocalDateTime now = LocalDateTime.now();
        List<SysTenantAppDO> codesToAdd = new ArrayList<>(appCode.size());
        SysTenantAppDO appDO = null;
        for (String s : appCode) {
            if (existsCodes.contains(s)) {
                continue;
            }
            appDO = new SysTenantAppDO();
            appDO.setAppCode(s);
            appDO.setSysTenantId(tenantId);
            appDO.setAssignTime(now);
            codesToAdd.add(appDO);
        }

        // 需要删除的
        Set<String> codesToDel = new HashSet<>(existsCodes.size());
        for (String s : existsCodes) {
            if (appCode.contains(s)) {
                continue;
            }
            codesToDel.add(s);
        }

        // 更新数据库
        if (!codesToAdd.isEmpty()) {
            tenantAppRepo.saveAll(codesToAdd);
        }
        if (!codesToDel.isEmpty()) {
            tenantAppRepoProc.deleteByTenantAndAppCode(tenantId, codesToDel);
        }

        // 清理租户缓存
        clearCache();
        return ApiResult.ok(tenantId);
    }

    @Override
    public ApiResult<SysTenantVO> getDetail(Long id) {
        var tenantDO = sysTenantRepoProc.get(id);
        if (tenantDO == null) {
            return ApiResult.fail("数据不存在");
        }

        var tenantVO = CONVERT.doToVO(tenantDO);
        tenantVO.setAdminUserId(tenantDO.getSysUserId());
        if (StringUtils.hasText(tenantVO.getIndustry())) {
            tenantVO.setIndustryName(getUdcIndustry().get(tenantVO.getIndustry()));
        }
        if (StringUtils.hasText(tenantVO.getCustomer())) {
            tenantVO.setCustomerName(getUdcCustomer().get(tenantVO.getCustomer()));
        }
        return ApiResult.ok(tenantVO);
    }

    @Override
    public ApiResult<Boolean> exists(Long id) {
        Assert.notNull(id, "租户ID为空");

        boolean exists = sysTenantRepoProc.exists(id);
        return ApiResult.ok(exists);
    }

    @Override
    public ApiResult<PagingVO<SysTenantVO>> search(SysTenantQueryParam param) {
        var queryResult = sysTenantRepoProc.query(param);
        if (!CollectionUtils.isEmpty(queryResult.getRecords())) {
            Map<String, String> industryMap = null;
            Map<String, String> customerMap = null;

            for (SysTenantVO tenantVO : queryResult.getRecords()) {
                if (CharSequenceUtil.isNotBlank(tenantVO.getIndustry())) {
                    if (industryMap == null) {
                        industryMap = this.getUdcIndustry();
                    }
                    tenantVO.setIndustryName(industryMap.get(tenantVO.getIndustry()));
                }
                if (CharSequenceUtil.isNotBlank(tenantVO.getCustomer())) {
                    if (customerMap != null) {
                        customerMap = this.getUdcIndustry();
                    }
                    tenantVO.setCustomerName(customerMap.get(tenantVO.getCustomer()));
                }
            }
        }

        return ApiResult.ok(queryResult);
    }

    @Override
    public ApiResult<List<SysUserTenantVO>> queryTenantOfUser(Long sysUserId) {
        var tenants = sysTenantUserRepoProc.queryUserTenant(sysUserId);
        return ApiResult.ok(tenants);
    }

    private void bindTenantUser(Long userId, Long tenantId) {
        var existsTenantIds = sysTenantUserRepoProc.getTenantIdOfUser(userId);
        if (existsTenantIds.contains(tenantId)) {
            return;
        }

        SysTenantUserDO tenantUserDO = new SysTenantUserDO();
        tenantUserDO.setSysUserId(userId);
        tenantUserDO.setSysTenantId(tenantId);
        tenantUserDO.setBindTime(LocalDateTime.now());
        tenantUserDO.setEnabled(true);

        sysTenantUserRepo.save(tenantUserDO);
    }

    private Long updateTenantAdmin(SysTenantUpdateParam updateParam, Long adminId) {
        if (!StringUtils.hasText(updateParam.getAdminAccount())) {
            // 未设置，则不修改
            return adminId;
        }
        Long newAdminId = userService.getUniqueIdByUsername(updateParam.getAdminAccount());
        if (newAdminId != null) {
            if (newAdminId.longValue() == adminId) {
                // 管理员未改变
                return adminId;
            }

            // 管理员属性调整
            addTenantAdminAttribute(newAdminId);
            removeTenantAdminAttribute(adminId);
            return newAdminId;
        }

        // 创建新的管理员
        newAdminId = addNewAdminUser(updateParam.getAdminAccount(), updateParam.getTenantName(), updateParam.getContactNumber());
        // 去掉老的管理员类型
        removeTenantAdminAttribute(adminId);

        return newAdminId;
    }

    private Long createTenantAdmin(SysTenantCreateParam createParam) {
        Long adminId = userService.getUniqueIdByUsername(createParam.getAdminAccount());
        if (adminId != null) {
            // 已存在，则更新
            addTenantAdminAttribute(adminId);
            return adminId;
        }

        return addNewAdminUser(createParam.getAdminAccount(), createParam.getTenantName(), createParam.getContactNumber());
    }

    private Long addNewAdminUser(String username, String tenantName, String mobile) {
        UserCreateDTO createDTO = new UserCreateDTO();
        createDTO.setUsername(username);
        createDTO.setFullName(tenantName + "管理员");
        createDTO.setGender(Gender.SECRET.getValue());
        createDTO.setMobile(mobile);
        createDTO.setEnabled(true);
        createDTO.setSourceType(UserSourceType.SYS.getValue());
        createDTO.setTerminalsAdd(Set.of(Terminal.BACKEND.name()));
        createDTO.setUserTypesAdd(Set.of(new SysUserTypeDTO(UserType.ADMIN_TENANT.getValue())));

        Long userId = null;
        try {
            userId = userService.insertUser(createDTO);
        } catch (Exception e) {
            throw new BusinessException("创建管理员账号失败：" + e.getMessage(), e);
        }

        // 解除与默认租户的关联
        sysTenantUserRepoProc.unbindTenant(TenantConstant.DEFAULT_TENANT_ID, userId);

        return userId;
    }

    private void addTenantAdminAttribute(Long adminId) {
        userService.addUserTerminal(adminId, Set.of(Terminal.BACKEND.name()));
        userService.addUserType(adminId, Set.of(UserType.ADMIN_TENANT.getValue()));
    }

    private void removeTenantAdminAttribute(Long adminId) {
        userService.removeUserType(adminId, Set.of(UserType.ADMIN_TENANT.getValue()));
    }

    private void checkForSave(SysTenantCreateParam createParam) {
        GeneralUserDetails currentUser = SecurityContextUtil.currentUserIfUnauthorizedThrow();
        Assert.isTrue(currentUser.isOperation(), "无权限操作");

        boolean exists = sysTenantRepoProc.existsTenantCode(createParam.getTenantCode(), null);
        Assert.isTrue(!exists, "编码已存在");

        createParam.setEnabled(ObjectUtil.defaultIfNull(createParam.getEnabled(), false));

        // 判断租户管理员账号是否存在
        exists = userService.existsUsername(createParam.getAdminAccount(), null);
        Assert.isTrue(!exists, "管理员账号已存在");

        if (StringUtils.hasText(createParam.getContactNumber())) {
            exists = userService.existsMobile(createParam.getContactNumber(), null);
            Assert.isTrue(!exists, "手机号已存在");
        }

        // 从配置中获取
        var isolation = tenantClientProperties.getIsolateStrategy();
//        var isolation = convertTenantIsolate(createParam.getTenantIsolation());
        if (isolation != null) {
            createParam.setTenantIsolation(isolation.name());
        }
        if (TenantIsolateStrategy.SCHEMA == isolation) {
            // 基于schema隔离
            Assert.hasText(createParam.getSchemaName(), "schema名称不能为空");
        } else if (TenantIsolateStrategy.FIELD == isolation) {
        } else {
            throw new IllegalArgumentException("不支持的数据隔离策略");
        }

        // 判断schemaName是否存在
        if (isolation == TenantIsolateStrategy.SCHEMA) {
            String schemaName = createParam.getSchemaName().toLowerCase();
            exists = sysTenantRepoProc.existsSchemaName(schemaName, null);
            Assert.isTrue(!exists, "schema名称已存在");
            Assert.isTrue(validateSchemaName(schemaName), "schema名称只能是小写字母和下划线组成");

            createParam.setSchemaName(schemaName);
        }

        // 检查域名
        if (CharSequenceUtil.isBlank(createParam.getTenantDomain())) {
            // 生成随机域名
            createParam.setTenantDomain(generateRandomDomain());
        } else {
            Assert.isTrue(validateChildDomain(createParam.getTenantDomain()), "域名格式不正确");

            exists = sysTenantRepoProc.existsTenantDomain(createParam.getTenantDomain(), null);
            Assert.isTrue(!exists, "域名已存在");
        }
        if (CharSequenceUtil.isNotBlank(createParam.getCustomDomain())) {
            var customDomain = normalizeDomain(createParam.getCustomDomain());
            Assert.isTrue(validateDomain(customDomain), "域名格式不正确");
            exists = sysTenantRepoProc.existsCustomDomain(createParam.getCustomDomain(), null);
            Assert.isTrue(!exists, "自定义域名已存在");

            createParam.setCustomDomain(customDomain);
        }
    }

    private void checkForUpdate(SysTenantUpdateParam updateParam, SysTenantDO tenantDOOld) {
        GeneralUserDetails currentUser = SecurityContextUtil.currentUserIfUnauthorizedThrow();
        Assert.isTrue(currentUser.isOperation(), "无权限操作");

        Assert.notNull(tenantDOOld, "修改的数据不存在");
        var id = updateParam.getId();

        boolean exists = sysTenantRepoProc.existsTenantCode(updateParam.getTenantCode(), id);
        Assert.isTrue(!exists, "编码已存在");

        updateParam.setEnabled(ObjectUtil.defaultIfNull(updateParam.getEnabled(), false));

        if (StringUtils.hasText(updateParam.getContactNumber()) && !updateParam.getContactNumber().equals(tenantDOOld.getContactNumber())) {
            exists = userService.existsMobile(updateParam.getContactNumber(), tenantDOOld.getSysUserId());
            Assert.isTrue(!exists, "手机号已存在");
        }

        // 从配置中获取
        var isolation = tenantClientProperties.getIsolateStrategy();
//        var isolation = SysTenantMngServiceImpl.convertTenantIsolate(updateParam.getTenantIsolation());
        if (isolation != null) {
            updateParam.setTenantIsolation(isolation.name());
        }
        if (TenantIsolateStrategy.SCHEMA == isolation) {
            Assert.hasText(updateParam.getSchemaName(), "schema名称不能为空");
        } else if (TenantIsolateStrategy.FIELD == isolation) {
        } else {
            throw new IllegalArgumentException("不支持的数据隔离策略");
        }

        // 检查数据库创建信息
        if (Boolean.TRUE.equals(tenantDOOld.getDbInitialized())) {
            TenantIsolateStrategy strategy = SysTenantMngServiceImpl.convertTenantIsolate(updateParam.getTenantIsolation());
            if (strategy == TenantIsolateStrategy.DATABASE) {
            } else if (strategy == TenantIsolateStrategy.SCHEMA) {
                Assert.isTrue(CharSequenceUtil.equals(updateParam.getSchemaName(), tenantDOOld.getSchemaName()), "数据库已初始化，不可修改schema");
            }
        }

        // 检查域名
        if (CharSequenceUtil.isNotBlank(updateParam.getTenantDomain()) && !CharSequenceUtil.equals(updateParam.getTenantDomain(), tenantDOOld.getTenantDomain())) {
            Assert.isTrue(validateChildDomain(updateParam.getTenantDomain()), "域名格式不正确");

            exists = sysTenantRepoProc.existsTenantDomain(updateParam.getTenantDomain(), id);
            Assert.isTrue(!exists, "域名已存在");
        }
        if (CharSequenceUtil.isNotBlank(updateParam.getCustomDomain()) && !CharSequenceUtil.equals(updateParam.getCustomDomain(), tenantDOOld.getCustomDomain())) {
            var customDomain = normalizeDomain(updateParam.getCustomDomain());
            Assert.isTrue(validateDomain(customDomain), "域名格式不正确");

            exists = sysTenantRepoProc.existsCustomDomain(updateParam.getCustomDomain(), id);
            Assert.isTrue(!exists, "自定义域名已存在");
        }
    }

    private void clearCache() {
        try {
            redisWrapper.apply(() -> {
                String cacheKey = TenantConstant.CACHE_ALL_KEY + ":all";
                redisUtils.del(cacheKey);
                return null;
            }, null);
        } catch (Exception e) {
            log.error("清理redis中租户缓存异常：", e);
        }
    }

    private boolean validateSchemaName(String schemaName) {
        String pattern = "[a-z|0-9|_]+";

        Pattern r = Pattern.compile(pattern);
        Matcher m = r.matcher(schemaName);
        return m.matches();
    }

    private boolean validateChildDomain(String childDomain) {
        String pattern = "[a-z|0-9|-]+";

        Pattern r = Pattern.compile(pattern);
        Matcher m = r.matcher(childDomain);
        return m.matches();
    }

    private boolean validateDomain(String domain) {
        String pattern = "[a-zA-Z0-9][-a-zA-Z0-9]{0,62}(\\.[a-zA-Z0-9][-a-zA-Z0-9]{0,62})+\\.?";
        Pattern r = Pattern.compile(pattern);
        var matcher = r.matcher(domain);
        return matcher.matches();
    }

    private String normalizeDomain(String domain) {
        domain = domain.toLowerCase();
        if (domain.startsWith(TenantConstant.PROTOCOL_HTTP)) {
            domain = domain.substring(7);
        } else if (domain.startsWith(TenantConstant.PROTOCOL_HTTPS)) {
            domain = domain.substring(8);
        }
        domain = domain.endsWith("/") ? domain.substring(0, domain.length() - 1) : domain;

        return domain;
    }

    private String generateRandomDomain() {
        while (true) {
            String domain = RandomUtil.randomString(9);
            if (sysTenantRepoProc.existsTenantDomain(domain, null)) {
                continue;
            }
            return domain;
        }
    }

    static TenantIsolateStrategy convertTenantIsolate(String isolate) {
        if (!StringUtils.hasText(isolate)) {
            return null;
        }

        return TenantIsolateStrategy.parse(isolate);
    }

    private Map<String, String> getUdcIndustry() {
        return super.udcMap(Application.NAME, "INDUSTRY");
    }

    private Map<String, String> getUdcCustomer() {
        return super.udcMap(Application.NAME, "CUSTOMER");
    }
}


