package com.elitescloud.cloudt.tenant.provider;

import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.boot.tenant.client.common.TenantClient;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import com.elitescloud.cloudt.system.param.SysUserNewParam;
import com.elitescloud.cloudt.system.spi.SysUserServiceSpi;
import com.elitescloud.cloudt.tenant.service.SysTenantMngService;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;

/**
 * 用户服务.
 *
 * @author Kaiser（wang shao）
 * 2022/4/6
 */
@Log4j2
public class SysUserServiceProvider implements SysUserServiceSpi {

    @Autowired(required = false)
    private SysTenantMngService tenantService;
    @Autowired
    private TenantClientProvider tenantClientProvider;

    @Override
    public String getServiceName() {
        return "租户管理";
    }

    @Override
    public void beforeUserAdd(SysUserNewParam userNewParam) {
        if (!tenantClientProvider.enabledTenant()) {
            // 未启用多租户
            return;
        }

        Long sysTenantId = userNewParam.getTenantId();
        if (sysTenantId == null) {
            return;
        }

        // 判断租户是否存在
        var result = tenantService.exists(sysTenantId);
        if (result == null || Boolean.FALSE.equals(result.getData())) {
            throw new BusinessException("租户不存在");
        }
    }

    @Override
    public void afterUserAdd(SysUserNewParam userNewParam, Long sysUserId) {
        if (!tenantClientProvider.enabledTenant()) {
            // 未启用多租户
            return;
        }

        Long sysTenantId = userNewParam.getTenantId();
        if (sysTenantId == null) {
            // 从当前用户中解析绑定

            GeneralUserDetails user = SecurityContextUtil.currentUser();
            // 自动绑定用户所属租户
            bindTenantUser(user, sysUserId);

            return;
        }

        tenantService.updateUserBindTenant(sysUserId, sysTenantId);
    }

    @Override
    public void afterUserUpdate(Long sysUserId) {
        if (!tenantClientProvider.enabledTenant()) {
            // 未启用多租户
            return;
        }

        GeneralUserDetails user = SecurityContextUtil.currentUser();

        // 自动绑定用户所属租户
        bindTenantUser(user, sysUserId);
    }

    @Override
    public void afterUserDelete(Long sysUserId) {
        if (!tenantClientProvider.enabledTenant()) {
            // 未启用多租户
            return;
        }

        GeneralUserDetails user = SecurityContextUtil.currentUser();

        // 自动绑定用户所属租户
        unbindTenantUser(user, sysUserId);
    }

    private void bindTenantUser(GeneralUserDetails currentUser, Long sysUserId) {
        if (sysUserId == null) {
            log.info("用户绑定租户失败，用户ID为空");
            return;
        }

        SysTenantDTO tenant = obtainCurrentTenant(currentUser);
        if (tenant == null) {
            log.info("没有获取到所属租户，无需绑定");
            return;
        }

        tenantService.updateUserBindTenant(sysUserId, tenant.getId());
    }

    private void unbindTenantUser(GeneralUserDetails currentUser, Long sysUserId) {
        if (currentUser == null) {
            return;
        }

        if (currentUser.isSystemAdmin() || currentUser.isOperation()) {
            // 运营机构时暂时不允许删除
            throw new BusinessException("请使用租户管理员删除租户");
        }

        // 租户时
        var tenant = obtainCurrentTenant(currentUser);
        if (tenant == null) {
            throw new BusinessException("未知当前用户所属租户");
        }
        tenantService.updateUserUnbindTenant(sysUserId, tenant.getId());
    }

    private SysTenantDTO obtainCurrentTenant(GeneralUserDetails currentUser) {
        SysTenantDTO tenant = ObjectUtil.defaultIfNull(TenantClient.getSessionTenant(), TenantClient.getCurrentTenant());
        if (tenant == null) {
            if (currentUser == null) {
                log.warn("当前用户不存在，租户上下文为空，绑定租户用户失败！");
                return null;
            }
            if (currentUser.isSystemAdmin() || currentUser.isOperation()) {
                return null;
            }

            tenant = currentUser.getTenant();
        }
        return tenant;
    }
}
