package com.elitescloud.cloudt.tenant.service.manager;

import cn.hutool.core.exceptions.ExceptionUtil;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.redis.util.RedisUtils;
import com.elitescloud.boot.support.app.CloudtAppHolder;
import com.elitescloud.boot.wrapper.RedisWrapper;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.constant.TenantIsolateStrategy;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.service.SysTenantBasicDataService;
import com.elitescloud.cloudt.system.service.model.entity.SysTenantDO;
import com.elitescloud.cloudt.tenant.common.BasicDataType;
import com.elitescloud.cloudt.system.config.TenantProperties;
import com.elitescloud.cloudt.tenant.model.entity.SysTenantBaseDataSyncDetailDO;
import com.elitescloud.cloudt.tenant.model.entity.SysTenantDbMigrateDO;
import com.elitescloud.cloudt.tenant.service.SysTenantDbMigrateService;
import com.elitescloud.cloudt.tenant.service.repo.SysTenantBaseDataSyncDetailRepo;
import com.elitescloud.cloudt.tenant.service.repo.SysTenantBaseDataSyncRepoProc;
import com.elitescloud.cloudt.tenant.service.repo.SysTenantRepoProc;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.task.TaskExecutor;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.time.LocalDateTime;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/3/23
 */
@Log4j2
public class SysTenantManager {

    @Autowired
    private SysTenantRepoProc sysTenantRepoProc;
    @Autowired
    private SysTenantBaseDataSyncRepoProc baseDataSyncRepoProc;
    @Autowired
    private SysTenantBaseDataSyncDetailRepo baseDataSyncDetailRepo;

    @Autowired
    private SysTenantBasicDataService tenantBasicDataService;
    @Autowired
    private SysTenantDbMigrateService tenantDbMigrateService;
    @Autowired
    private TenantProperties tenantProperties;
    @Autowired
    private TaskExecutor taskExecutor;
    @Autowired
    private TenantClientCaller tenantClientCaller;
    @Autowired
    private RedisWrapper redisWrapper;
    @Autowired
    private RedisUtils redisUtils;

    /**
     * 更新客户端数据库ddl
     *
     * @param tenantDO 租户信息
     * @return 更新结果
     */
    public boolean syncClientDb(SysTenantDO tenantDO) {
        if (!CharSequenceUtil.equals(tenantDO.getTenantIsolation(), TenantIsolateStrategy.SCHEMA.name())
                && !CharSequenceUtil.equals(tenantDO.getTenantIsolation(), TenantIsolateStrategy.DATABASE.name())) {
            // 无需创建db
            return false;
        }

        if (Boolean.TRUE.equals(tenantProperties.isAsyncCall())) {
            // 异步更新
            CompletableFuture.supplyAsync(() -> executeSyncClientDb(tenantDO), taskExecutor)
                    .whenComplete((res, e) -> {
                        if (e != null) {
                            log.error("更新客户端租户schema异常：", e);
                            return;
                        }
                        log.info("更新客户端租户【{}, {}】schema成功！", tenantDO.getSchemaName(), tenantDO.getId());
                        SpringContextHolder.getBean(SysTenantManager.class).updateTenantDbInitialized(tenantDO.getId(), true);
                    });
            return true;
        }
        // 同步更新
        return executeSyncClientDb(tenantDO);
    }

    /**
     * 重新执行指定的记录
     *
     * @param migrateDO
     * @return
     */
    public boolean retryDbMigrate(SysTenantDbMigrateDO migrateDO) {
        syncTargetClientDb(migrateDO.getId(), migrateDO.getAppCode(), migrateDO.getSchemaName());

        return true;
    }

    /**
     * 删除客户端数据库
     *
     * @param tenantDO 租户信息
     * @return 删除结果
     */
    public boolean deleteClientDb(SysTenantDO tenantDO) {
        if (!CharSequenceUtil.equals(tenantDO.getTenantIsolation(), TenantIsolateStrategy.SCHEMA.name())
                && !CharSequenceUtil.equals(tenantDO.getTenantIsolation(), TenantIsolateStrategy.DATABASE.name())) {
            // 无需删除db
            return false;
        }

        CompletableFuture.supplyAsync(() -> executeDeleteClientDb(tenantDO), taskExecutor)
                .whenComplete((res, e) -> {
                    if (e != null) {
                        log.error("删除客户端租户schema异常：", e);
                    }
                    log.info("删除客户端租户【{}, {}】成功！", tenantDO.getSchemaName(), tenantDO.getId());
                });
        return true;
    }

    /**
     * 同步租户基础数据
     *
     * @param syncId
     * @param tenantId
     */
    @TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
    @TenantOrgTransaction(useTenantOrg = false)
    public void syncBaseData(long syncId, long tenantId) {
        // 更新同步开始时间
        baseDataSyncRepoProc.updateStarted(syncId);

        String failReason = "";
        Throwable exception = null;
        boolean success = true;
        try {
            // 开始同步UDC
            syncBaseData(tenantId, syncId, BasicDataType.UDC);
            // 开始同步发号器
            syncBaseData(tenantId, syncId, BasicDataType.SEQUENCE);

            log.info("同步租户【" + tenantId + "】基础数据成功");
        } catch (Throwable e) {
            log.error("同步租户【" + tenantId + "】基础数据异常：", e);
            failReason = ExceptionUtil.stacktraceToString(e, -1);
            exception = e;
            success = false;
        }

        // 更新同步结果
        try {
            baseDataSyncRepoProc.updateEnd(syncId, success, failReason);
        } catch (Exception e) {
            log.error("更新同步结果异常：", e);
            if (exception == null) {
                // 正常结束
                return;
            }
            throw new RuntimeException(exception);
        }
    }

    @TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
    @TenantOrgTransaction(useTenantOrg = false)
    public void updateSyncResult(long syncId, Throwable throwable) {
        if (throwable == null) {
            // 更新成功
            baseDataSyncRepoProc.updateEnd(syncId, true, null);
            return;
        }

        // 更新失败
        String failReason = ExceptionUtil.stacktraceToString(throwable, -1);
        baseDataSyncRepoProc.updateEnd(syncId, false, failReason);
    }

    /**
     * 更新是否已初始化
     *
     * @param sysTenantId
     * @param initizlized
     */
    @TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
    @TenantOrgTransaction(useTenantOrg = false)
    public void updateTenantDbInitialized(Long sysTenantId, boolean initizlized) {
        sysTenantRepoProc.updateDbInitialized(sysTenantId, initizlized);

        clearTenantCache();
    }

    private void syncBaseData(Long tenantId, Long syncId, BasicDataType dataType) {
        // 保存同步记录明细
        SysTenantBaseDataSyncDetailDO detailDO = new SysTenantBaseDataSyncDetailDO();
        detailDO.setSyncId(syncId);
        detailDO.setDataType(dataType.getValue());
        detailDO.setFinished(false);
        detailDO.setSuccess(false);
        detailDO.setStartTime(LocalDateTime.now());
        baseDataSyncDetailRepo.save(detailDO);

        // 开始同步
        boolean success = true;
        String failReason = "";
        try {
            ApiResult<Boolean> syncResult = null;
            if (dataType == BasicDataType.UDC) {
                syncResult = tenantBasicDataService.syncUdc(tenantId);
            } else if (dataType == BasicDataType.SEQUENCE) {
                syncResult = tenantBasicDataService.syncSequence(tenantId);
            } else {
                syncResult = ApiResult.fail("暂不支持的数据类型");
            }
            Assert.isTrue(syncResult.isSuccess(), syncResult.getMsg());
        } catch (Exception e) {
            log.error("同步租户【" + tenantId + "】数据【" + dataType + "】异常：", e);
            success = false;
            failReason = ExceptionUtil.stacktraceToString(e, -1);
        } finally {
            // 更新同步结果
            detailDO.setSuccess(success);
            detailDO.setFinished(true);
            detailDO.setEndTime(LocalDateTime.now());
            detailDO.setFailReason(failReason);

            baseDataSyncDetailRepo.save(detailDO);
        }
    }

    private boolean executeDeleteClientDb(SysTenantDO tenantDO) {
        // 获取客户端编码
        Set<String> apps = obtainApps();
        if (CollectionUtils.isEmpty(apps)) {
            throw new BusinessException("未获取到有效的租户客户端");
        }

        for (String app : apps) {
            // 调用删除
            Long migrateId = tenantDbMigrateService.getMigrateId(tenantDO.getTenantId(), app).getData();
            if (migrateId == null) {
                continue;
            }
            deleteTargetClientDb(migrateId, app, tenantDO.getSchemaName());
        }

        // 更新租户的数据库创建状态
        sysTenantRepoProc.updateDbInitialized(tenantDO.getId(), false);
        return true;
    }

    private Set<String> obtainApps() {
        Set<String> apps = tenantProperties.getClientNames();
        if (tenantProperties.isDiscoveryClient() && CollectionUtils.isEmpty(apps)) {
            apps = tenantClientCaller.allAppCodes();
        }

        apps = apps == null ? new HashSet<>(4) : new HashSet<>(apps);

        var currentAppName = CloudtAppHolder.getAppCode();
        if (CharSequenceUtil.isNotBlank(currentAppName)) {
            apps.add(currentAppName);
        }

        return apps;
    }

    private boolean executeSyncClientDb(SysTenantDO tenantDO) {
        TenantIsolateStrategy isolateStrategy = TenantIsolateStrategy.parse(tenantDO.getTenantIsolation());

        // 获取客户端编码
        Set<String> apps = obtainApps();

        // 循环更新对应的客户端应用
        for (String app : apps) {
            // 保存更新记录
            Long migrateId = tenantDbMigrateService.addInit(tenantDO, isolateStrategy, app).getData();
            syncTargetClientDb(migrateId, app, tenantDO.getSchemaName());
        }

        return true;
    }

    private void deleteTargetClientDb(Long migrateId, String appCode, String schemaName) {
        boolean success = false;
        try {
            tenantClientCaller.callDatabaseRpcProvider(appCode, tenantProperties.getCreateTimeOut(), rpcProvider -> {
                ApiResult<Boolean> deleteResult = rpcProvider.schemaDrop(schemaName);
                if (deleteResult != null && deleteResult.isSuccess()) {
                    log.info("删除租户客户端schema成功：{}，{}", appCode, schemaName);
                    return true;
                }
                return false;
            });
            success = true;
        } catch (Exception e) {
            log.error("删除租户客户端schema异常：", e);
        }

        // 更新迁移结果
        if (success) {
            tenantDbMigrateService.updateResult(migrateId, false, "已删除");
        }
    }

    private void syncTargetClientDb(Long migrateId, String appCode, String schemaName) {
        String lastError = null;
        boolean success = false;
        try {
            tenantClientCaller.callDatabaseRpcProvider(appCode, tenantProperties.getCreateTimeOut(), rpcProvider -> {
                ApiResult<Boolean> createResult = rpcProvider.schemaCreate(schemaName);
                if (createResult != null && createResult.isSuccess()) {
                    return true;
                }
                return false;
            });
            success = true;
        } catch (Exception e) {
            lastError = e.getMessage();
        }

        if (success) {
            // 处理成功，更新结果
            tenantDbMigrateService.updateResult(migrateId, true, null);
        } else {
            tenantDbMigrateService.updateResult(migrateId, false, lastError);
        }
    }

    public void clearTenantCache() {
        try {
            redisWrapper.apply(() -> {
                String cacheKey = TenantConstant.CACHE_ALL_KEY + ":all";
                redisUtils.del(cacheKey);
                return null;
            }, null);
        } catch (Exception e) {
            log.error("清理redis中租户缓存异常：", e);
        }
    }
}
