package com.elitescloud.cloudt.tenant.service.impl;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.service.repo.SysTenantAppRepoProc;
import com.elitescloud.cloudt.tenant.convert.SysTenantDatasourceConvert;
import com.elitescloud.cloudt.tenant.model.entity.SysTenantDatasourceDO;
import com.elitescloud.cloudt.tenant.model.vo.params.SysTenantDatasourceRespVO;
import com.elitescloud.cloudt.tenant.model.vo.params.SysTenantDatasourceSaveVO;
import com.elitescloud.cloudt.tenant.service.SysTenantDatasourceService;
import com.elitescloud.cloudt.tenant.service.repo.SysTenantDatasourceRepoProc;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2023/9/27
 */
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
@TenantOrgTransaction(useTenantOrg = false)
public class SysTenantDatasourceServiceImpl extends BaseServiceImpl implements SysTenantDatasourceService {
    private static final CodeNameParam APP_DEFAULT = new CodeNameParam("default", "默认");
    private static final String DRIVER_DEF = "net.sf.log4jdbc.sql.jdbcapi.DriverSpy";

    @Autowired
    private SysTenantDatasourceRepoProc repoProc;
    @Autowired
    private SysTenantAppRepoProc tenantAppRepoProc;

    @Override
    public ApiResult<List<CodeNameParam>> appList(Long sysTenantId) {
        sysTenantId = this.normalizeTenantId(sysTenantId);
        if (sysTenantId == null) {
            return ApiResult.fail("租户ID为空");
        }

        var tenantAppList = tenantAppRepoProc.getAppOfTenant(sysTenantId);

        List<CodeNameParam> appList = new ArrayList<>(16);
        appList.add(APP_DEFAULT);
        appList.addAll(tenantAppList);

        return ApiResult.ok(appList);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> save(SysTenantDatasourceSaveVO saveVO) {
        var datasourceDO = this.saveVo2Do(saveVO);
        repoProc.save(datasourceDO);

        return ApiResult.ok(datasourceDO.getId());
    }

    @Override
    public ApiResult<String> testConnection(SysTenantDatasourceSaveVO saveVO) {
        saveVO.setSysTenantId(this.normalizeTenantId(saveVO.getSysTenantId()));
        Assert.notNull(saveVO.getSysTenantId(), "租户ID为空");
        Assert.notBlank(saveVO.getAppCode(), "应用编码为空");

        String failMsg = null;
        if (Boolean.TRUE.equals(saveVO.getUseDef())) {
            // 验证默认数据源
            var defaultDatasource = repoProc.getByTenantAndApp(saveVO.getSysTenantId(), saveVO.getAppCode());
            if (defaultDatasource == null) {
                return ApiResult.fail("请先配置默认数据源");
            }
            failMsg = this.connectionDatasource(defaultDatasource.getDsUrl(), defaultDatasource.getDsUsername(), defaultDatasource.getDsPassword(), defaultDatasource.getDsDriver(), defaultDatasource.getDsPoolProp());
        } else {
            failMsg = this.connectionDatasource(saveVO.getDsUrl(), saveVO.getDsUsername(), saveVO.getDsPassword(), saveVO.getDsDriver(), saveVO.getDsPoolProp());
        }

        if (CharSequenceUtil.isBlank(failMsg)) {
            return ApiResult.ok("数据源有效，可使用");
        }
        return ApiResult.fail("数据源无效，连接失败：" + failMsg);
    }

    @Override
    public ApiResult<SysTenantDatasourceRespVO> get(Long sysTenantId, String appCode) {
        sysTenantId = this.normalizeTenantId(sysTenantId);
        if (sysTenantId == null) {
            return ApiResult.fail("租户ID为空");
        }
        if (CharSequenceUtil.isBlank(appCode)) {
            return ApiResult.fail("应用编码为空");
        }

        var datasourceDO = repoProc.getByTenantAndApp(sysTenantId, appCode);
        if (datasourceDO == null) {
            return ApiResult.ok();
        }

        var respVO = SysTenantDatasourceConvert.INSTANCE.do2RespVO(datasourceDO);
        return ApiResult.ok(respVO);
    }

    private String connectionDatasource(String url, String username, String password, String driver, String prop) {
        return null;
    }

    private SysTenantDatasourceDO saveVo2Do(SysTenantDatasourceSaveVO saveVO) {
        saveVO.setSysTenantId(this.normalizeTenantId(saveVO.getSysTenantId()));
        Assert.notNull(saveVO.getSysTenantId(), "租户ID为空");
        Assert.notBlank(saveVO.getAppCode(), "应用编码为空");

        saveVO.setEnabled(ObjectUtil.defaultIfNull(saveVO.getEnabled(), false));
        saveVO.setUseDef(ObjectUtil.defaultIfNull(saveVO.getUseDef(), false));

        if (Boolean.FALSE.equals(saveVO.getUseDef())) {
            // 校验数据源配置
            this.validateDatasourceConfig(saveVO);
        }

        var datasourceDO = repoProc.getByTenantAndApp(saveVO.getSysTenantId(), saveVO.getAppCode());
        if (datasourceDO == null) {
            datasourceDO = SysTenantDatasourceConvert.INSTANCE.saveVo2Do(saveVO);
        } else {
            SysTenantDatasourceConvert.INSTANCE.copySaveVo2Do(saveVO, datasourceDO);
        }

        datasourceDO.setDef(APP_DEFAULT.getCode().equals(saveVO.getAppCode()));

        return datasourceDO;
    }

    private void validateDatasourceConfig(SysTenantDatasourceSaveVO saveVO) {
        // 校验数据源配置
        Assert.notBlank(saveVO.getDsUrl(), "数据源url为空");
        Assert.notBlank(saveVO.getDsUsername(), "数据源账号为空");
        Assert.notBlank(saveVO.getDsPassword(), "数据源密码为空");
        saveVO.setDsDriver(CharSequenceUtil.blankToDefault(saveVO.getDsDriver(), DRIVER_DEF));
    }

    private Long normalizeTenantId(Long sysTenantId) {
        if (sysTenantId != null && sysTenantId.longValue() != TenantConstant.DEFAULT_TENANT_ID) {
            return sysTenantId;
        }
        var tenant = super.currentTenant();
        if (tenant == null) {
            return null;
        }
        return tenant.getId();
    }
}
