package com.elitescloud.cloudt.tenant.service.manager.impl;

import com.alibaba.nacos.api.exception.NacosException;
import com.alibaba.nacos.api.naming.pojo.Instance;
import com.elitescloud.boot.dubbo.util.DubboNacosHolder;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.tenant.client.common.TenantDatabaseRpcProvider;
import com.elitescloud.cloudt.tenant.service.manager.TenantClientCaller;
import lombok.extern.log4j.Log4j2;
import org.apache.dubbo.common.constants.RegistryConstants;
import org.springframework.lang.NonNull;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/8/1
 */
@Log4j2
public class DubboTenantClientCaller implements TenantClientCaller {

    @Override
    public Set<String> allAppCodes() {
        Class<TenantDatabaseRpcProvider> clientInterface = TenantDatabaseRpcProvider.class;
        return queryClientInstance(clientInterface).keySet();
    }

    @Override
    public boolean callDatabaseRpcProvider(@NonNull String appCode, Integer timeout, @NonNull Predicate<TenantDatabaseRpcProvider> rpcProviderFunction) {
        // 先查询客户端实例
        Class<TenantDatabaseRpcProvider> clientInterface = TenantDatabaseRpcProvider.class;
        Map<String, List<Instance>> clientInstanceMap = queryClientInstance(clientInterface);
        List<Instance> instances = clientInstanceMap.get(appCode);
        if (instances.isEmpty()) {
            throw new BusinessException("调用租户客户端时未查询到有效客户端");
        }

        // 循环调用客户端应用
        TenantDatabaseRpcProvider provider = null;
        String lastError = null;
        for (Instance instance : instances) {
            // 获取客户端实例
            try {
                provider = DubboNacosHolder.getInstanceOfSpecial(clientInterface, null, instance, timeout);
            } catch (Exception e) {
                log.error("获取租户客户端异常：", e);
                lastError = "获取租户客户端失败：" + e.getMessage();
                continue;
            }

            // 执行业务逻辑
            boolean result;
            try {
                result = rpcProviderFunction.test(provider);
            } catch (Exception e) {
                lastError = "租户客户端执行失败：" + e.getMessage();
                continue;
            }

            // 一个应用集群里只要一个实例更新成功即可
            if (result) {
                log.info("调用租户客户端接口成功：{}", appCode);
                return true;
            }
        }

        throw new BusinessException(lastError);
    }

    private Map<String, List<Instance>> queryClientInstance(Class<?> clientInterface) {
        String serviceName = DubboNacosHolder.buildServiceName(RegistryConstants.PROVIDERS_CATEGORY, clientInterface, null);

        // 查询所有实例
        Map<String, List<Instance>> instanceMap = null;
        try {
            instanceMap = DubboNacosHolder.queryNacosInstanceForApplication(serviceName, null);
        } catch (NacosException e) {
            throw new BusinessException("查询租户的业务客户端失败", e);
        }
        if (instanceMap.isEmpty()) {
            return instanceMap;
        }

        // 过滤出健康实例
        Map<String, List<Instance>> resultMap = new HashMap<>(instanceMap.size());
        List<Instance> instances = null;
        for (Map.Entry<String, List<Instance>> entry : instanceMap.entrySet()) {
            instances = entry.getValue().stream()
                    .filter(t -> t.isEnabled() && t.isHealthy())
                    .collect(Collectors.toList());
            if (instances.isEmpty()) {
                continue;
            }

            resultMap.put(entry.getKey(), instances);
        }
        return resultMap;
    }
}
