package com.elitesland.workflow;

import com.elitescloud.cloudt.workflow.Application;
import com.elitescloud.cloudt.workflow.params.ProcDefDTO;
import com.elitescloud.cloudt.workflow.params.TaskNodeConfigDTO;
import com.elitesland.workflow.payload.*;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;

import javax.validation.constraints.NotBlank;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * <p>
 * 工作流服务
 * <p>
 * 这是工作流提供的服务,使用方法
 *
 * @author yangzhibin
 * @DubboReference private WorkflowService workflowService;
 * <p>
 * workflowService.startProcess(ProcDefKey.LEAVE.name(),"12356789",null);//启动工作流
 *
 * </p >
 * @date 2020/8/19
 */
@FeignClient(name = Application.NAME, path = WorkflowService.URI)
public interface WorkflowService {
    String URI = Application.URI_PREFIX + "/wf";

    @GetMapping(value = "/hello")
    WorkflowResult<String> hello(@RequestParam(name = "username", required = false) String username);

    /**
     * 启动工作流
     *
     * @param payload 启动参数
     * @return 返回流程信息
     */
    @PostMapping("/workflow/api/startProcess")
    WorkflowResult<ProcessInfo> startProcess(@RequestBody StartProcessPayload payload);


    /**
     * 设置流程变量
     *
     * @param payload 流程变量参数
     */
    @PostMapping("/workflow/api/setVariables")
    WorkflowResult<Object> setVariables(@RequestBody SetVariablesPayload payload);

    /**
     * 查询流程当前任务信息
     *
     * @param payload 流程实例ID
     * @return 当前任务信息
     */
    @PostMapping("/workflow/api/currentTaskInfos")
    WorkflowResult<HashMap<String, TaskInfo>> currentTaskInfos(@RequestBody CurrentTaskInfosPayload payload);


    /**
     * 根据流程实例ID,查询流程当前任务信息
     *
     * @param procInstId 流程实例ID
     * @return 当前任务信息
     */
    @PostMapping("/workflow/api/currentTaskInfo")
    WorkflowResult<TaskInfo> currentTaskInfo(@RequestParam(name = "procInstId") String procInstId);


    /**
     * 根据流程实例ID,查询流程审批信息
     *
     * @param procInstId 流程实例ID
     * @return 该流程审批信息
     */
    @PostMapping("/workflow/api/commentInfos")
    WorkflowResult<ArrayList<CommentInfo>> commentInfos(@RequestParam(name = "procInstId") String procInstId);


    /**
     * 删除流程,不回调业务
     *
     * @param payload 参数
     */
    @PostMapping("/workflow/api/deleteProcess")
    WorkflowResult<Object> deleteProcess(@RequestBody DeleteProcessPayload payload);


    /**
     * 可驳回的节点
     */
    @GetMapping("/workflow/api/canBackNodes")
    WorkflowResult<Object> canBackNodes(@RequestParam(name = "procInstId") String procInstId);

    /**
     * 审批[同意/拒绝]
     */
    @PostMapping("/workflow/api/approve")
    WorkflowResult<Object> approve(@RequestBody @Validated ApprovePayload payload);

    /**
     * 获取节点配置
     *
     * @param procDefKey 流程定义key
     * @param taskDefId  任务节点id，为空则是流程的所有配置节点
     * @return 节点配置列表
     */
    @GetMapping(value = "/workflow/api/taskNodeConfig")
    WorkflowResult<List<TaskNodeConfigDTO>> getNodeConfig(@RequestParam(name = "procDefKey") @NotBlank(message = "流程定义Key为空") String procDefKey,
                                                          @RequestParam(name = "taskDefId", required = false) String taskDefId);

    /**
     * 获取流程定义信息
     *
     * @param procDefKey 流程定义Key
     * @return 流程定义信息
     */
    @GetMapping(value = "/workflow/api/procDef")
    WorkflowResult<ProcDefDTO> getProcDef(@RequestParam(name = "procDefKey") @NotBlank(message = "流程定义Key为空") String procDefKey);

    /**
     * 获取流程定义的任务节点
     *
     * @param procDefKey
     * @return
     */
    @GetMapping(value = "/workflow/api/userTaskNames")
    WorkflowResult<Map<String, String>> getUserTaskNames(@RequestParam(name = "procDefKey") @NotBlank(message = "流程定义Key为空") String procDefKey);
}