package com.elitesland.yst.common.base.param;

import com.elitesland.yst.common.base.ApiCode;
import com.elitesland.yst.common.exception.BusinessException;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.querydsl.core.types.Order;
import com.querydsl.core.types.OrderSpecifier;
import com.querydsl.core.types.dsl.EntityPathBase;
import com.querydsl.core.types.dsl.PathBuilder;
import com.querydsl.core.types.dsl.PathBuilderValidator;
import com.querydsl.jpa.impl.JPAQuery;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.val;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 可排序查询参数对象
 *
 * @author Michael Li
 * @date 2019-08-04
 */
@Data
@EqualsAndHashCode(callSuper = true)
@ApiModel("可排序查询参数对象")
public abstract class AbstractOrderQueryParam extends QueryParam {
    private static final long serialVersionUID = 57714391204790143L;

    /**
     * 排序
     */
    @ApiModelProperty(value = "排序")
    private List<OrderItem> orders;

    public List<OrderItem> getOrders() {
        return orders;
    }

    public void setOrders(List<OrderItem> orders) {
        this.orders = orders;
    }

    public void defaultOrder(OrderItem orderItem) {
        this.defaultOrders(Collections.singletonList(orderItem));
    }

    public void defaultOrders(List<OrderItem> orderItems) {
        if (orderItems == null || orderItems.isEmpty()) {
            return;
        }
        this.orders = orderItems;
    }

    @JsonIgnore
    public PageRequest getPageRequest() {
        val orderBys = CollectionUtils.isEmpty(orders) ? new ArrayList<Sort.Order>() :
                orders.stream().map(o -> new Sort.Order(
                        o.isAsc() ? Sort.Direction.ASC : Sort.Direction.DESC,
                        o.getColumn()
                )).collect(Collectors.toList());

        return PageRequest.of(getCurrent(), getSize(), Sort.by(orderBys));
    }

    /**
     * 单表查询,排序方法
     * @param query
     * @param entityPathBase 实体对象
     */
    @JsonIgnore
    public void fillOrders(JPAQuery<?> query, EntityPathBase<?> entityPathBase) {
        val pageRequest = getPageRequest();
        val orderbyExpr = new PathBuilder<>(entityPathBase.getClass(),
                entityPathBase.getMetadata(), PathBuilderValidator.FIELDS);
        pageRequest.getSort().forEach(s -> query.orderBy(
                new OrderSpecifier(
                        s.isAscending() ? Order.ASC : Order.DESC,
                        orderbyExpr.get(s.getProperty())
                )
        ));
    }

    /**
     * 多表关联查询，排序方法。要求：排序字段格式需为 “实体对象名称.字段名称”
     * @param query
     */
    @JsonIgnore
    public void fillOrders(JPAQuery<?> query) {
        val pageRequest = getPageRequest();
        pageRequest.getSort().forEach(s -> {
            val sects = s.getProperty().split("\\.");
            if (sects.length > 1) {
                val orderbyExpr = new PathBuilder<>(Object.class, sects[0]);
                query.orderBy(new OrderSpecifier(s.isAscending() ?
                        Order.ASC : Order.DESC, orderbyExpr.get(sects[1])));
            } else {
                throw new BusinessException(ApiCode.PARAMETER_PARSE_EXCEPTION, "排序字段需要指定所属hql的实体对象，对象名称.字段名称");
            }
        });
    }

    @JsonIgnore
    public void setPaging(JPAQuery<?> query) {
        query.offset((long) getCurrent() * getSize());
        query.limit(getSize());
    }
}
