package com.elitesland.boot.autoconfigure.elasticsearch.config;

import cn.hutool.core.lang.TypeReference;
import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.ArrayUtil;
import com.elitesland.boot.autoconfigure.elasticsearch.annotation.EnableElasticsearch;
import com.elitesland.boot.elasticsearch.config.ElasticsearchClientRegister;
import com.elitesland.boot.elasticsearch.support.ElasticsearchInit;
import lombok.extern.slf4j.Slf4j;
import org.elasticsearch.client.Node;
import org.elasticsearch.client.RestClient;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.BeanFactoryAware;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.boot.autoconfigure.AutoConfigurationPackages;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.data.elasticsearch.ReactiveElasticsearchRestClientProperties;
import org.springframework.boot.autoconfigure.elasticsearch.ElasticsearchRestClientProperties;
import org.springframework.boot.autoconfigure.elasticsearch.RestClientBuilderCustomizer;
import org.springframework.boot.context.properties.PropertyMapper;
import org.springframework.context.annotation.*;
import org.springframework.core.type.AnnotationMetadata;
import org.springframework.data.elasticsearch.client.ClientConfiguration;
import org.springframework.data.elasticsearch.core.ElasticsearchRestTemplate;
import org.springframework.http.HttpHeaders;
import org.springframework.lang.NonNull;
import org.springframework.util.unit.DataSize;

import java.net.InetSocketAddress;
import java.net.URI;

/**
 * Elasticsearch配置入口.
 *
 * @author Kaiser（wang shao）
 * @date 2021/7/1
 */
@ConditionalOnClass(value = ElasticsearchRestTemplate.class)
@Import({ElasticsearchCanalClientConfiguration.class, ElasticsearchReactiveClientConfiguration.class})
@Role(BeanDefinition.ROLE_INFRASTRUCTURE)
@Slf4j
public class ElasticsearchConfiguration implements ImportAware, BeanFactoryAware, ApplicationRunner {

    private BeanFactory beanFactory;
    private AnnotationMetadata importMetadata;

    @Bean
    @Role(BeanDefinition.ROLE_INFRASTRUCTURE)
    public RestClientBuilderCustomizer restClientBuilderCustomizer() {
        return builder -> builder.setFailureListener(new RestClient.FailureListener() {
            @Override
            public void onFailure(Node node) {
                logger.error("Elasticsearch 失败节点：{}", node);
            }
        });
    }

    @Override
    public void setBeanFactory(@NonNull BeanFactory beanFactory) throws BeansException {
        this.beanFactory = beanFactory;
    }

    @Override
    public void setImportMetadata(@NonNull AnnotationMetadata importMetadata) {
        this.importMetadata = importMetadata;
    }

    @Override
    public void run(ApplicationArguments args) throws Exception {
        String[] packages = getBasePackages(this.importMetadata, this.beanFactory);
        var elasticsearchRestTemplate = beanFactory.getBean(ElasticsearchRestTemplate.class);
        new ElasticsearchInit(elasticsearchRestTemplate, packages).execute();
    }

    private static String[] getBasePackages(AnnotationMetadata importingClassMetadata, BeanFactory beanFactory) {
        var annotationConfig = importingClassMetadata.getAnnotationAttributes(EnableElasticsearch.class.getName());
        if (annotationConfig == null) {
            throw new IllegalArgumentException("请使用EnableElasticsearch注解开启Elasticsearch的配置");
        }

        String[] packages = MapUtil.get(annotationConfig, "basePackages", new TypeReference<>() {
        });

        if (ArrayUtil.isEmpty(packages)) {
            if (beanFactory == null || !AutoConfigurationPackages.has(beanFactory)) {
                logger.warn("Elasticsearch Client Service 自动注册失败，没有找到有效的BeanFactory ！");
                return packages;
            }
            packages = AutoConfigurationPackages.get(beanFactory).toArray(String[]::new);
        }
        return packages;
    }

    @Configuration(proxyBeanMethods = false)
    @ConditionalOnMissingBean(ElasticsearchClientRegister.class)
    @Role(BeanDefinition.ROLE_INFRASTRUCTURE)
    public static class OnConfigFailConfiguration implements InitializingBean {

        @Override
        public void afterPropertiesSet() throws Exception {
            logger.info("Elasticsearch 扩展未成功配置");
        }
    }

    @Role(BeanDefinition.ROLE_INFRASTRUCTURE)
    public static class AutoConfigureElasticsearchClientRegister implements ImportBeanDefinitionRegistrar, BeanFactoryAware {

        private BeanFactory beanFactory;

        @Override
        public void setBeanFactory(@NonNull BeanFactory beanFactory) throws BeansException {
            this.beanFactory = beanFactory;
        }

        @Override
        public void registerBeanDefinitions(@NonNull AnnotationMetadata importingClassMetadata, @NonNull BeanDefinitionRegistry registry) {
            String[] packages = getBasePackages(importingClassMetadata, this.beanFactory);
            if (ArrayUtil.isEmpty(packages)) {
                logger.info("未获取到有效的Elasticsearch设置的相关包路径，请确保{}设置正确", EnableElasticsearch.class);
                return;
            }

            var beanDefinition = BeanDefinitionBuilder.genericBeanDefinition(ElasticsearchClientRegister.class)
                    .addConstructorArgValue(packages)
                    .getRawBeanDefinition();
            registry.registerBeanDefinition("autoConfigureElasticsearchClientRegister", beanDefinition);
        }
    }
}
