package com.elitesland.boot.autoconfigure.elasticsearch.config;

import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.elasticsearch.ElasticsearchProperties;
import org.springframework.boot.context.properties.PropertyMapper;
import org.springframework.context.annotation.Bean;
import org.springframework.data.elasticsearch.client.ClientConfiguration;
import org.springframework.data.elasticsearch.client.reactive.ReactiveRestClients;
import org.springframework.http.HttpHeaders;
import org.springframework.util.unit.DataSize;
import org.springframework.web.reactive.function.client.ExchangeStrategies;
import org.springframework.web.reactive.function.client.WebClient;

import java.net.InetSocketAddress;
import java.net.URI;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2021/11/18
 */
@ConditionalOnClass({ReactiveRestClients.class, WebClient.class})
class ElasticsearchReactiveClientConfiguration {

    @Bean
    public ClientConfiguration clientConfiguration(ElasticsearchProperties restClientProperties) {
        InetSocketAddress[] endpoints = restClientProperties.getUris().stream()
                .map(URI::create)
                .map(uri -> new InetSocketAddress(uri.getHost(), uri.getPort()))
                .toArray(InetSocketAddress[]::new);
        ClientConfiguration.MaybeSecureClientConfigurationBuilder builder = ClientConfiguration.builder()
                .connectedTo(endpoints);

        boolean useSsl = restClientProperties.getUris().stream().anyMatch(t -> t.toLowerCase().startsWith("https"));
        if (useSsl) {
            builder.usingSsl();
        }
        configureTimeouts(builder, restClientProperties);
        configureExchangeStrategies(builder, restClientProperties);
        return builder.build();
    }

    private void configureTimeouts(ClientConfiguration.TerminalClientConfigurationBuilder builder,
                                   ElasticsearchProperties properties) {
        PropertyMapper map = PropertyMapper.get();
        map.from(properties.getConnectionTimeout()).whenNonNull().to(builder::withConnectTimeout);
        map.from(properties.getSocketTimeout()).whenNonNull().to(builder::withSocketTimeout);
        map.from(properties.getUsername()).whenHasText().to(username -> {
            HttpHeaders headers = new HttpHeaders();
            headers.setBasicAuth(username, properties.getPassword());
            builder.withDefaultHeaders(headers);
        });
    }

    private void configureExchangeStrategies(ClientConfiguration.TerminalClientConfigurationBuilder builder,
                                             ElasticsearchProperties properties) {
        PropertyMapper map = PropertyMapper.get();
        builder.withWebClientConfigurer(webClient -> {
            ExchangeStrategies exchangeStrategies = ExchangeStrategies.builder()
                    .codecs(configurer -> map.from(DataSize.ofKilobytes(10)).whenNonNull()
                            .asInt(DataSize::toBytes)
                            .to(maxInMemorySize -> configurer.defaultCodecs().maxInMemorySize(maxInMemorySize)))
                    .build();
            return webClient.mutate().exchangeStrategies(exchangeStrategies).build();
        });
    }
}
