package com.elitesland.boot.elasticsearch;

import com.elitesland.boot.elasticsearch.common.BaseDocument;
import com.elitesland.boot.elasticsearch.common.PageResult;
import com.elitesland.boot.elasticsearch.common.query.AggregationParam;
import com.elitesland.boot.elasticsearch.common.query.ElasticsearchQueryHelper;
import com.elitesland.boot.elasticsearch.common.query.QueryParam;
import com.elitesland.boot.elasticsearch.common.query.SearchParam;
import org.elasticsearch.search.aggregations.Aggregations;

import java.util.List;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2021/6/24
 */
public interface ElasticsearchClientService<T extends BaseDocument, E> {

    /**
     * 获取索引的名称
     *
     * @return 索引名称
     */
    String getIndexName();

    /**
     * 判断索引是否存在
     *
     * @return 是否存在
     */
    boolean indexExists();

    /**
     * 索引创建
     * <p>
     * 如果索引存在，直接返回，否则创建
     *
     * @return 是否成功
     */
    boolean indexCreate();

    /**
     * 索引更新
     * <p>
     * 如果索引存在，则更新索引，否则直接创建
     *
     * @return 是否成功
     */
    boolean indexUpdate();

    /**
     * 索引删除
     * <p>
     * 如果索引下有文档，会一起删除
     *
     * @return 是否成功
     */
    boolean indexDelete();

    /**
     * 文档保存
     * <p>
     * 若已存在，则更新
     *
     * @param document 文档
     * @return 文档
     */
    T documentSave(T document);

    /**
     * 文档批量保存
     * <p>
     * 存在的则更新
     *
     * @param documentList 文档列表
     * @return 文档列表
     */
    List<T> documentSave(List<T> documentList);

    /**
     * 判断文档是否已存在
     *
     * @param id 文档的标识
     * @return 是否存在
     */
    boolean documentExists(E id);

    /**
     * 文档获取
     *
     * @param id 文档的标识
     * @return 文档
     */
    T documentGet(E id);

    /**
     * 文档获取
     *
     * @param idList 文档的标识
     * @return 文档
     */
    List<T> documentGet(List<E> idList);

    /**
     * 文档删除
     *
     * @param id 文档的标识
     * @return 是否成功
     */
    boolean documentDelete(E id);

    /**
     * 文档批量删除
     *
     * @param idList 文档的标识
     * @return 是否成功
     */
    boolean documentDelete(List<E> idList);

    /**
     * 查询数据
     * <p>
     * 根据条件过滤出符合条件的数据，然后按照指定的顺序排序；
     * query.
     *
     * @param queryParam 搜索条件，请使用{@link ElasticsearchQueryHelper}
     * @return 搜索结果
     */
    PageResult<T> documentQuery(QueryParam<T> queryParam);

    /**
     * 检索数据
     * <p>
     * 根据条件过滤出符合条件的数据，然后通过查询条件进行计算相关性得分，依据得分进行排序；
     *
     * @param searchParam 搜索条件，请使用{@link ElasticsearchQueryHelper}
     * @return 搜索结果
     */
    PageResult<T> documentSearch(SearchParam<T> searchParam);

    /**
     * 聚合查询
     * <p>
     * 根据条件过滤出符合条件的数据，然后进行聚合运算；
     *
     * @param aggregationParam 参数，请使用{@link ElasticsearchQueryHelper}
     * @return 聚合结果
     */
    Aggregations aggregationSearch(AggregationParam<T> aggregationParam);

    /**
     * 自动补全
     *
     * @param prefix 待补全的前缀
     * @param size   返回数据量，默认10
     * @return 补全的内容列表
     */
    List<String> suggestComplete(String prefix, Integer size);
}
