package com.elitesland.boot.elasticsearch.canal.factory;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.StrUtil;
import com.alibaba.otter.canal.client.CanalConnector;
import com.alibaba.otter.canal.client.CanalConnectors;
import com.alibaba.otter.canal.protocol.CanalEntry;
import com.elitesland.boot.elasticsearch.CanalClient;
import com.elitesland.boot.elasticsearch.canal.config.CanalProperties;
import com.elitesland.boot.elasticsearch.canal.config.support.CanalHandlerCustomizer;
import com.elitesland.boot.elasticsearch.canal.model.RowData;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.http.converter.json.Jackson2ObjectMapperBuilder;
import org.springframework.util.StopWatch;

import java.io.Serializable;
import java.lang.reflect.ParameterizedType;
import java.net.InetSocketAddress;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 客户端工厂类.
 *
 * @author Kaiser（wang shao）
 * @date 2021/07/20
 */
@Slf4j
public class CanalClientFactory {

    private final BeanFactory beanFactory;
    private final CanalProperties canalProperties;
    private Map<String, Set<CanalClient>> canalClients;
    private Map<CanalClient, Class<? extends Serializable>> clientType;
    private ObjectMapper objectMapper;
    private CanalHandlerCustomizer canalHandlerCustomizer;

    public CanalClientFactory(BeanFactory beanFactory, CanalProperties canalProperties) {
        this.beanFactory = beanFactory;
        this.canalProperties = canalProperties;
    }

    public void build() {
        var watch = new StopWatch();
        logger.info("start build Elasticsearch Canal Client factory...");
        watch.start();

        var clients = getClients();
        if (clients.isEmpty()) {
            logger.info("not found available Canal Client");
            watch.stop();
            return;
        }

        buildObjectMapper();

        registerCanalClient(clients);
        CompletableFuture.runAsync(this::registerListener);

        watch.stop();
        logger.info("finish build Elasticsearch Canal Client factory in {}ms", watch.getTotalTimeMillis());
    }

    private void buildObjectMapper() {
        var builder = Jackson2ObjectMapperBuilder.json();
        canalHandlerCustomizer = beanFactory.getBean(CanalHandlerCustomizer.class);
        canalHandlerCustomizer.objectMapperBuilder(builder);

        this.objectMapper = builder.build();
    }

    private void registerListener() {
        CanalConnector connector = getConnector();
        loadData(connector);
    }

    private void loadData(CanalConnector connector) {
        while (true) {
            try {
                var message = connector.getWithoutAck(canalProperties.getBatchSize());
                var batchId = message.getId();

                if (batchId != -1 && message.getEntries().size() > 0) {
                    try {
                        dispatch(message.getEntries());
                        // 数据处理成功，则确认
                        connector.ack(batchId);
                    } catch (Exception exception) {
                        logger.error("数据处理失败", exception);
                        // 数据处理失败则回滚
                        connector.rollback(batchId);
                    }
                }
            } catch (Exception e) {
                logger.error("Elasticsearch Canal 加载数据失败", e);
            }
            try {
                TimeUnit.SECONDS.sleep(canalProperties.getPeriodRefresh().getSeconds());
            } catch (InterruptedException interruptedException) {
                logger.error("sleep异常", interruptedException);
            }
        }
    }

    private void dispatch(List<CanalEntry.Entry> entries) {
        String key;
        Set<CanalClient> clients;
        for (CanalEntry.Entry entry : entries) {
            if (entry.getEntryType() == CanalEntry.EntryType.TRANSACTIONBEGIN || entry.getEntryType() == CanalEntry.EntryType.TRANSACTIONEND) {
                continue;
            }
            key = entry.getHeader().getSchemaName() + "." + entry.getHeader().getTableName();
            clients = canalClients.get(key);
            if (CollUtil.isEmpty(clients)) {
                // 不存在对应的服务处理
                continue;
            }

            // 解析和处理数据
            CanalEntry.RowChange rowChange = null;
            try {
                rowChange = CanalEntry.RowChange.parseFrom(entry.getStoreValue());
            } catch (Exception e) {
                throw new RuntimeException("Elasticsearch Canal 解析数据异常, data:" + entry, e);
            }

            var eventType = rowChange.getEventType();
            for (CanalEntry.RowData rowData : rowChange.getRowDatasList()) {
                process(clients, eventType, rowData);
            }
        }
    }

    private void process(Set<CanalClient> clients, CanalEntry.EventType eventType, CanalEntry.RowData canalRowData) {
        if (eventType == CanalEntry.EventType.DELETE) {
            RowData rowData = convertRow(canalRowData.getBeforeColumnsList());
            for (var client : clients) {
                rowData.setData(objectMapper.convertValue(rowData.getValueMap(), clientType.get(client)));

                client.onDelete(rowData);
            }
            return;
        }
        if (eventType == CanalEntry.EventType.INSERT) {
            RowData rowData = convertRow(canalRowData.getAfterColumnsList());
            for (var client : clients) {
                rowData.setData(objectMapper.convertValue(rowData.getValueMap(), clientType.get(client)));

                client.onInsert(rowData);
            }
            return;
        }

        RowData rowDataBefore = convertRow(canalRowData.getBeforeColumnsList());
        RowData rowDataAfter = convertRow(canalRowData.getAfterColumnsList());
        for (var client : clients) {
            rowDataBefore.setData(objectMapper.convertValue(rowDataBefore.getValueMap(), clientType.get(client)));
            rowDataAfter.setData(objectMapper.convertValue(rowDataAfter.getValueMap(), clientType.get(client)));

            client.onUpdate(rowDataBefore, rowDataAfter);
        }
    }

    private RowData convertRow(List<CanalEntry.Column> canalColumns) {
        RowData rowData = new RowData();
        int size = canalColumns.size();
        List<RowData.Column> columnList = new ArrayList<>(size);
        Map<String, Serializable> valueMap = new HashMap<>(size);
        rowData.setColumnList(columnList);
        rowData.setValueMap(valueMap);
        if (size == 0) {
            return rowData;
        }

        RowData.Column column = null;
        for (CanalEntry.Column col : canalColumns) {
            column = new RowData.Column();
            column.setName(col.getName());
            column.setValue(col.getValue());

            columnList.add(column);
            valueMap.put(canalHandlerCustomizer.fieldNameConvert(col.getName()), col.getValue());
        }

        return rowData;
    }

    private CanalConnector getConnector() {
        CanalConnector connector = null;
        while (true) {
            try {
                connector = createConnector();
                if (!connector.checkValid()) {
                    logger.warn("canal server 不可用");
                    connector = null;
                }
            } catch (Exception e) {
                logger.error("创建Canal Connector失败", e);
            }
            if (connector == null) {
                try {
                    TimeUnit.SECONDS.sleep(5);
                } catch (InterruptedException e) {
                    logger.error("sleep 中断", e);
                }
                continue;
            }

            return connector;
        }
    }

    private CanalConnector createConnector() {
        String host = Assert.notBlank(canalProperties.getServer().getHost(), "Canal Server的配置host为空");
        String destination = Assert.notBlank(canalProperties.getServer().getDestination(), "Canal Server的配置destination为空");
        String username = StrUtil.nullToDefault(canalProperties.getServer().getUsername(), "");
        String password = StrUtil.nullToDefault(canalProperties.getServer().getPassword(), "");

        var connector = CanalConnectors.newSingleConnector(new InetSocketAddress(host,
                canalProperties.getServer().getPort()), destination, username, password);

        connector.connect();
        connector.subscribe(".*\\..*");
        connector.rollback();
        return connector;
    }

    private void registerCanalClient(Set<CanalClient> clients) {
        int size = clients.size();
        this.canalClients = new HashMap<>(size);
        this.clientType = new HashMap<>(size);

        String key;
        for (var client : clients) {
            key = StrUtil.blankToDefault(client.database(), canalProperties.getDatabase());
            key = (StrUtil.isBlank(key) ? "" : key + ".") + Assert.notBlank(client.table(), "{}中的table为空", client.getClass().getName());
            canalClients.computeIfAbsent(key, k -> new HashSet<>(4)).add(client);

            clientType.put(client, (Class<? extends Serializable>) ((ParameterizedType) client.getClass().getGenericInterfaces()[0]).getActualTypeArguments()[0]);

            logger.info("register Elasticsearch Canal Client 【{}】 listen to 【{}】", client.getClass().getName(), key);
        }
    }

    private Set<CanalClient> getClients() {
        return beanFactory.getBeanProvider(CanalClient.class).stream().collect(Collectors.toSet());
    }

}
