package com.elitesland.boot.elasticsearch.common.query;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.StrUtil;
import com.elitesland.boot.elasticsearch.common.BaseDocument;
import com.elitesland.boot.elasticsearch.support.DocumentIndexCache;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.index.query.RangeQueryBuilder;
import org.springframework.data.elasticsearch.annotations.FieldType;

import java.util.HashSet;
import java.util.Set;

/**
 * 查询条件构造器.
 *
 * @author Kaiser（wang shao）
 * @date 2021/07/06
 */
public class ConditionBuilder<T extends BaseDocument> {

    private final BoolQueryBuilder queryBuilder;
    private final Class<T> documentType;
    private final DocumentIndexCache.DocumentIndex documentIndex;
    private final QueryParamValueConvert paramValueConvert = new QueryParamValueConvert();

    ConditionBuilder(Class<T> clazz) {
        this.queryBuilder = QueryBuilders.boolQuery();
        documentType = clazz;
        documentIndex = DocumentIndexCache.getDocumentIndex(clazz);
        Assert.notNull(documentIndex, "构建查询条件失败");
    }

    /**
     * and field = value
     *
     * @param field 属性名
     * @param value 属性值
     * @return 条件构造器
     */
    public ConditionBuilder<T> and(String field, Object value) {
        return and(true, field, value, true);
    }

    /**
     * and field = value
     *
     * @param field 属性名
     * @param value 属性值
     * @param match 条件，如果为true则追加该条件
     * @return 条件构造器
     */
    public ConditionBuilder<T> and(String field, Object value, boolean match) {
        return and(match, field, value, true);
    }

    /**
     * and field > value
     *
     * @param field 属性名
     * @param value 属性值
     * @return 条件构造器
     */
    public ConditionBuilder<T> andGt(String field, Object value) {
        return gt(true, field, value, false, true);
    }

    /**
     * and field > value
     *
     * @param field 属性名
     * @param value 属性值
     * @param match 条件，如果为true则追加该条件
     * @return 条件构造器
     */
    public ConditionBuilder<T> andGt(String field, Object value, boolean match) {
        return gt(match, field, value, false, true);
    }

    /**
     * and field >= value
     *
     * @param field 属性名
     * @param value 属性值
     * @return 条件构造器
     */
    public ConditionBuilder<T> andGte(String field, Object value) {
        return gt(true, field, value, true, true);
    }

    /**
     * and field >= value
     *
     * @param field 属性名
     * @param value 属性值
     * @param match 条件，如果为true则追加该条件
     * @return 条件构造器
     */
    public ConditionBuilder<T> andGte(String field, Object value, boolean match) {
        return gt(match, field, value, true, true);
    }

    /**
     * and field < value
     *
     * @param field 属性名
     * @param value 属性值
     * @return 条件构造器
     */
    public ConditionBuilder<T> andLt(String field, Object value) {
        return lt(true, field, value, false, true);
    }

    /**
     * and field < value
     *
     * @param field 属性名
     * @param value 属性值
     * @param match 条件，如果为true则追加该条件
     * @return 条件构造器
     */
    public ConditionBuilder<T> andLt(String field, Object value, boolean match) {
        return lt(match, field, value, false, true);
    }

    /**
     * and field <= value
     *
     * @param field 属性名
     * @param value 属性值
     * @return 条件构造器
     */
    public ConditionBuilder<T> andLte(String field, Object value) {
        return lt(true, field, value, true, true);
    }

    /**
     * and field <= value
     *
     * @param field 属性名
     * @param value 属性值
     * @param match 条件，如果为true则追加该条件
     * @return 条件构造器
     */
    public ConditionBuilder<T> andLte(String field, Object value, boolean match) {
        return lt(match, field, value, true, true);
    }

    /**
     * and between from to
     *
     * @param field       属性名
     * @param from        from
     * @param includeFrom 是否包含from
     * @param to          to
     * @param includeTo   是否包含to
     * @return 条件构造器
     */
    public ConditionBuilder<T> andBetween(String field, Object from, boolean includeFrom, Object to, boolean includeTo) {
        return between(true, field, from, includeFrom, to, includeTo, true);
    }

    /**
     * and between from to
     *
     * @param field       属性名
     * @param from        from
     * @param includeFrom 是否包含from
     * @param to          to
     * @param includeTo   是否包含to
     * @param match       条件，true时追加该条件
     * @return 条件构造器
     */
    public ConditionBuilder<T> andBetween(String field, Object from, boolean includeFrom, Object to, boolean includeTo, boolean match) {
        return between(match, field, from, includeFrom, to, includeTo, true);
    }

    /**
     * 条件拼接
     *
     * @param conditionBuilder 条件构造器
     * @return 条件构造器
     */
    public ConditionBuilder<T> and(ConditionBuilder<T> conditionBuilder) {
        return and(conditionBuilder, true);
    }

    /**
     * 条件拼接
     *
     * @param conditionBuilder 条件构造器
     * @param match            满足则追加条件
     * @return 条件构造器
     */
    public ConditionBuilder<T> and(ConditionBuilder<T> conditionBuilder, boolean match) {
        if (!match) {
            return this;
        }
        queryBuilder.must(conditionBuilder.build());
        return this;
    }

    /**
     * and field != value
     *
     * @param field 属性名
     * @param value 属性值
     * @return 条件构造器
     */
    public ConditionBuilder<T> andNot(String field, Object value) {
        return and(true, field, value, false);
    }

    /**
     * and field != value
     *
     * @param field 属性名
     * @param value 属性值
     * @param match 条件，如果为true则追加该条件
     * @return 条件构造器
     */
    public ConditionBuilder<T> andNot(String field, Object value, boolean match) {
        return and(match, field, value, false);
    }

    /**
     * 条件拼接
     *
     * @param conditionBuilder 条件构造器
     * @return 条件构造器
     */
    public ConditionBuilder<T> andNotEqual(ConditionBuilder<T> conditionBuilder) {
        return andNotEqual(conditionBuilder, true);
    }

    /**
     * 条件拼接
     *
     * @param conditionBuilder 条件构造器
     * @param match            条件，如果为true则追加该条件
     * @return 条件构造器
     */
    public ConditionBuilder<T> andNotEqual(ConditionBuilder<T> conditionBuilder, boolean match) {
        if (!match) {
            return this;
        }
        queryBuilder.mustNot(conditionBuilder.build());
        return this;
    }

    /**
     * or field = value
     *
     * @param field 属性名
     * @param value 属性值
     * @return 条件构造器
     */
    public ConditionBuilder<T> or(String field, Object value) {
        return or(true, field, value);
    }

    /**
     * or field = value
     *
     * @param field 属性名
     * @param value 属性值
     * @param match 条件，如果为true则追加该条件
     * @return 条件构造器
     */
    public ConditionBuilder<T> or(String field, Object value, boolean match) {
        return or(match, field, value);
    }

    /**
     * or field > value
     *
     * @param field 属性名
     * @param value 属性值
     * @return 条件构造器
     */
    public ConditionBuilder<T> orGt(String field, Object value) {
        return gt(true, field, value, false, false);
    }

    /**
     * or field > value
     *
     * @param field 属性名
     * @param value 属性值
     * @param match 条件，如果为true则追加该条件
     * @return 条件构造器
     */
    public ConditionBuilder<T> orGt(String field, Object value, boolean match) {
        return gt(match, field, value, false, false);
    }

    /**
     * or field >= value
     *
     * @param field 属性名
     * @param value 属性值
     * @return 条件构造器
     */
    public ConditionBuilder<T> orGte(String field, Object value) {
        return gt(true, field, value, true, false);
    }

    /**
     * or field >= value
     *
     * @param field 属性名
     * @param value 属性值
     * @param match 条件，如果为true则追加该条件
     * @return 条件构造器
     */
    public ConditionBuilder<T> orGte(String field, Object value, boolean match) {
        return gt(match, field, value, true, false);
    }

    /**
     * or field < value
     *
     * @param field 属性名
     * @param value 属性值
     * @return 条件构造器
     */
    public ConditionBuilder<T> orLt(String field, Object value) {
        return lt(true, field, value, false, false);
    }

    /**
     * or field < value
     *
     * @param field 属性名
     * @param value 属性值
     * @param match 条件，如果为true则追加该条件
     * @return 条件构造器
     */
    public ConditionBuilder<T> orLt(String field, Object value, boolean match) {
        return lt(match, field, value, false, false);
    }

    /**
     * or field <= value
     *
     * @param field 属性名
     * @param value 属性值
     * @return 条件构造器
     */
    public ConditionBuilder<T> orLte(String field, Object value) {
        return lt(true, field, value, true, false);
    }

    /**
     * or field <= value
     *
     * @param field 属性名
     * @param value 属性值
     * @param match 条件，如果为true则追加该条件
     * @return 条件构造器
     */
    public ConditionBuilder<T> orLte(String field, Object value, boolean match) {
        return lt(match, field, value, true, false);
    }

    /**
     * or between from to
     *
     * @param field       属性名
     * @param from        from
     * @param includeFrom 是否包含from
     * @param to          to
     * @param includeTo   是否包含to
     * @return 条件构造器
     */
    public ConditionBuilder<T> orBetween(String field, Object from, boolean includeFrom, Object to, boolean includeTo) {
        return between(true, field, from, includeFrom, to, includeTo, false);
    }

    /**
     * or between from to
     *
     * @param field       属性名
     * @param from        from
     * @param includeFrom 是否包含from
     * @param to          to
     * @param includeTo   是否包含to
     * @param match       条件，如果为true则追加该条件
     * @return 条件构造器
     */
    public ConditionBuilder<T> orBetween(String field, Object from, boolean includeFrom, Object to, boolean includeTo, boolean match) {
        return between(match, field, from, includeFrom, to, includeTo, false);
    }

    /**
     * 条件拼接
     *
     * @param conditionBuilder 条件构造器
     * @return 条件构造器
     */
    public ConditionBuilder<T> or(ConditionBuilder<T> conditionBuilder) {
        return or(conditionBuilder, true);
    }

    /**
     * 条件拼接
     *
     * @param conditionBuilder 条件构造器
     * @param match            条件，如果为true则追加该条件
     * @return 条件构造器
     */
    public ConditionBuilder<T> or(ConditionBuilder<T> conditionBuilder, boolean match) {
        if (!match) {
            return this;
        }
        queryBuilder.should(conditionBuilder.build());
        return this;
    }

    public QueryBuilder build() {
        return queryBuilder;
    }

    private ConditionBuilder<T> and(boolean match, String field, Object value, boolean equal) {
        if (!match || StrUtil.isBlank(field)) {
            return this;
        }

        QueryBuilder builder = supportTerm(field, value) ? QueryBuilders.termQuery(field, convertValue(field, value)) : QueryBuilders.matchQuery(field, value);
        if (equal) {
            queryBuilder.must(builder);
        } else {
            queryBuilder.mustNot(builder);
        }

        return this;
    }

    /**
     * or field = value
     *
     * @param match 条件，如果为true则追加该条件
     * @param field 属性名
     * @param value 属性值
     * @return 条件构造器
     */
    public ConditionBuilder<T> or(boolean match, String field, Object value) {
        if (!match || StrUtil.isBlank(field)) {
            return this;
        }

        QueryBuilder builder = supportTerm(field, value) ? QueryBuilders.termQuery(field, convertValue(field, value)) : QueryBuilders.matchQuery(field, value);
        queryBuilder.should(builder);

        return this;
    }

    private ConditionBuilder<T> gt(boolean match, String field, Object value, boolean include, boolean and) {
        if (!match || StrUtil.isBlank(field) || value == null) {
            return this;
        }

        RangeQueryBuilder builder = QueryBuilders.rangeQuery(field);
        if (include) {
            builder.gt(convertValue(field, value));
        } else {
            builder.gte(convertValue(field, value));
        }

        if (and) {
            queryBuilder.must(builder);
        } else {
            queryBuilder.should(builder);
        }

        return this;
    }

    private ConditionBuilder<T> lt(boolean match, String field, Object value, boolean include, boolean and) {
        if (!match || StrUtil.isBlank(field) || value == null) {
            return this;
        }

        RangeQueryBuilder builder = QueryBuilders.rangeQuery(field);
        if (include) {
            builder.lt(convertValue(field, value));
        } else {
            builder.lte(convertValue(field, value));
        }

        if (and) {
            queryBuilder.must(builder);
        } else {
            queryBuilder.should(builder);
        }

        return this;
    }

    public ConditionBuilder<T> between(boolean match, String field, Object from, boolean includeFrom, Object to, boolean includeTo, boolean and) {
        if (!match || StrUtil.isBlank(field) || (from == null && to == null)) {
            return this;
        }

        RangeQueryBuilder builder = QueryBuilders.rangeQuery(field);
        if (from != null) {
            if (includeFrom) {
                builder.gte(convertValue(field, from));
            } else {
                builder.gt(convertValue(field, from));
            }
        }
        if (to != null) {
            if (includeTo) {
                builder.lte(convertValue(field, to));
            } else {
                builder.lt(convertValue(field, to));
            }
        }

        if (and) {
            queryBuilder.must(builder);
        } else {
            queryBuilder.should(builder);
        }

        return this;
    }

    /**
     * 是否支持精确查询
     *
     * @param property 属性名
     * @param value    属性值
     * @return 是否支持
     */
    private boolean supportTerm(String property, Object value) {
        var exists = documentIndex.getFields().contains(property);
        Assert.isTrue(exists, "在{}中未找到有效的属性{}", documentType.getName(), property);

        var fieldWrapper = documentIndex.getFieldWrapperMap().get(property);
        var field = fieldWrapper == null ? null : fieldWrapper.getElasticsearchField();
        Assert.notNull(field, "在{}中{}不是有效的索引字段", documentType.getName(), property);

        exists = FIELD_TYPE_SUPPORT_TERM.contains(field.type());
        if (exists) {
            return true;
        }
        return value == null || value.toString().length() < field.ignoreAbove();
    }

    private Object convertValue(String property, Object value) {
        if (value == null) {
            return null;
        }

        var field = documentIndex.getFieldWrapperMap().get(property);
        Assert.notNull(field, "未知属性{}", property);
        return paramValueConvert.convert(field, value);
    }

    private static final Set<FieldType> FIELD_TYPE_SUPPORT_TERM = new HashSet<>(64);

    static {
        FIELD_TYPE_SUPPORT_TERM.add(FieldType.Keyword);
        FIELD_TYPE_SUPPORT_TERM.add(FieldType.Long);
        FIELD_TYPE_SUPPORT_TERM.add(FieldType.Integer);
        FIELD_TYPE_SUPPORT_TERM.add(FieldType.Short);
        FIELD_TYPE_SUPPORT_TERM.add(FieldType.Byte);
        FIELD_TYPE_SUPPORT_TERM.add(FieldType.Double);
        FIELD_TYPE_SUPPORT_TERM.add(FieldType.Float);
        FIELD_TYPE_SUPPORT_TERM.add(FieldType.Half_Float);
        FIELD_TYPE_SUPPORT_TERM.add(FieldType.Scaled_Float);
        FIELD_TYPE_SUPPORT_TERM.add(FieldType.Date);
        FIELD_TYPE_SUPPORT_TERM.add(FieldType.Date_Nanos);
        FIELD_TYPE_SUPPORT_TERM.add(FieldType.Boolean);
    }
}
