package com.elitesland.boot.elasticsearch.common.query;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ArrayUtil;
import com.elitesland.boot.elasticsearch.common.BaseDocument;
import com.elitesland.boot.elasticsearch.support.DocumentIndexCache;
import org.elasticsearch.search.sort.FieldSortBuilder;
import org.elasticsearch.search.sort.SortBuilders;
import org.elasticsearch.search.sort.SortOrder;
import org.springframework.data.elasticsearch.annotations.Field;
import org.springframework.data.elasticsearch.annotations.FieldType;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * 排序构造器.
 *
 * @author Kaiser（wang shao）
 * @date 2021/07/06
 */
public class OrderBuilder<T extends BaseDocument> {

    private final List<FieldSortBuilder> sortBuilders;
    private final Class<T> documentType;
    private final DocumentIndexCache.DocumentIndex documentIndex;

    private OrderBuilder(Class<T> documentType) {
        sortBuilders = new ArrayList<>(32);
        this.documentType = documentType;
        this.documentIndex = DocumentIndexCache.getDocumentIndex(documentType);
    }

    /**
     * 创建顺序构造器实例
     *
     * @return 顺序构造器
     */
    public static <T extends BaseDocument> OrderBuilder<T> instance(Class<T> documentClass) {
        return new OrderBuilder<>(documentClass);
    }

    /**
     * 设置排序
     *
     * @param order  排序方式
     * @param fields 字段名称
     * @return 构造器
     */
    public OrderBuilder<T> order(Order order, String... fields) {
        if (ArrayUtil.isEmpty(fields)) {
            return null;
        }

        for (String f : fields) {
            Assert.isTrue(support(f), "{}不是有效的排序字段", f);

            sortBuilders.add(SortBuilders.fieldSort(f).order(order == Order.ASC ? SortOrder.ASC : SortOrder.DESC));
        }

        return this;
    }

    public List<FieldSortBuilder> build() {
        return this.sortBuilders;
    }

    private boolean support(String property) {
        var exists = documentIndex.getFields().contains(property);
        Assert.isTrue(exists, "在{}中未找到有效的属性{}", documentType.getName(), property);
        var fieldWrapper = documentIndex.getFieldWrapperMap().get(property);
        var field = fieldWrapper == null ? null : fieldWrapper.getElasticsearchField();
        Assert.notNull(field, "在{}中{}不是有效的索引字段", documentType.getName(), property);

        return FIELD_TYPE_SUPPORT_ORDER.contains(field.type());
    }

    private static final Set<FieldType> FIELD_TYPE_SUPPORT_ORDER = new HashSet<>(64);

    static {
        FIELD_TYPE_SUPPORT_ORDER.add(FieldType.Keyword);
        FIELD_TYPE_SUPPORT_ORDER.add(FieldType.Long);
        FIELD_TYPE_SUPPORT_ORDER.add(FieldType.Integer);
        FIELD_TYPE_SUPPORT_ORDER.add(FieldType.Short);
        FIELD_TYPE_SUPPORT_ORDER.add(FieldType.Byte);
        FIELD_TYPE_SUPPORT_ORDER.add(FieldType.Double);
        FIELD_TYPE_SUPPORT_ORDER.add(FieldType.Float);
        FIELD_TYPE_SUPPORT_ORDER.add(FieldType.Half_Float);
        FIELD_TYPE_SUPPORT_ORDER.add(FieldType.Scaled_Float);
        FIELD_TYPE_SUPPORT_ORDER.add(FieldType.Date);
        FIELD_TYPE_SUPPORT_ORDER.add(FieldType.Date_Nanos);
        FIELD_TYPE_SUPPORT_ORDER.add(FieldType.Boolean);
    }
}
