package com.elitesland.boot.elasticsearch.common.query;

import cn.hutool.core.lang.Assert;
import com.elitesland.boot.elasticsearch.support.FieldWrapper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.elasticsearch.annotations.DateFormat;

import java.text.SimpleDateFormat;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAccessor;
import java.util.Date;
import java.util.EnumMap;
import java.util.Map;

/**
 * 查询参数值转换.
 *
 * @author Kaiser（wang shao）
 * @date 2021/10/12
 */
@Slf4j
public class QueryParamValueConvert {

    private static final Map<DateFormat, String> DATE_FORMAT = new EnumMap<>(DateFormat.class);

    static {
        DATE_FORMAT.put(DateFormat.basic_date, "yyyyMMdd");
        DATE_FORMAT.put(DateFormat.basic_date_time, "yyyyMMdd'T'HHmmss.SSSZ");
        DATE_FORMAT.put(DateFormat.basic_date_time_no_millis, "yyyyDDD'T'HHmmssZ");
        DATE_FORMAT.put(DateFormat.basic_ordinal_date, "yyyyDDD");
        DATE_FORMAT.put(DateFormat.basic_ordinal_date_time, "yyyyDDD'T'HHmmss.SSSZ");
        DATE_FORMAT.put(DateFormat.basic_ordinal_date_time_no_millis, "yyyyDDD'T'HHmmssZ");
        DATE_FORMAT.put(DateFormat.basic_time, "HHmmss.SSSZ");
        DATE_FORMAT.put(DateFormat.basic_time_no_millis, "HHmmssZ");
        DATE_FORMAT.put(DateFormat.basic_t_time, "'T'HHmmss.SSSZ");
        DATE_FORMAT.put(DateFormat.basic_t_time_no_millis, "'T'HHmmssZ");
        DATE_FORMAT.put(DateFormat.basic_week_date, "xxxx'W'wwe");
        DATE_FORMAT.put(DateFormat.basic_week_date_time, "xxxx'W'wwe'T'HHmmss.SSSZ");
        DATE_FORMAT.put(DateFormat.basic_week_date_time_no_millis, "xxxx'W'wwe'T'HHmmssZ");
        DATE_FORMAT.put(DateFormat.date, "yyyy-MM-dd");
        DATE_FORMAT.put(DateFormat.date_hour, "yyyy-MM-dd'T'HH");
        DATE_FORMAT.put(DateFormat.date_hour_minute, "yyyy-MM-dd'T'HH:mm");
        DATE_FORMAT.put(DateFormat.date_hour_minute_second, "yyyy-MM-dd'T'HH:mm:ss");
        DATE_FORMAT.put(DateFormat.date_hour_minute_second_fraction, "yyyy-MM-dd'T'HH:mm:ss.SSS");
        DATE_FORMAT.put(DateFormat.date_hour_minute_second_millis, "yyyy-MM-dd'T'HH:mm:ss.SSS");
        DATE_FORMAT.put(DateFormat.date_optional_time, "yyyy-MM-dd'T'HH:mm:ss.SSSZ");
        DATE_FORMAT.put(DateFormat.date_time, "yyyy-MM-dd'T'HH:mm:ss.SSSZZ");
        DATE_FORMAT.put(DateFormat.date_time_no_millis, "yyyy-MM-dd'T'HH:mm:ssZZ");
        DATE_FORMAT.put(DateFormat.hour, "HH");
        DATE_FORMAT.put(DateFormat.hour_minute, "HH:mm");
        DATE_FORMAT.put(DateFormat.hour_minute_second, "HH:mm:ss");
        DATE_FORMAT.put(DateFormat.hour_minute_second_fraction, "HH:mm:ss.SSS");
        DATE_FORMAT.put(DateFormat.hour_minute_second_millis, "HH:mm:ss.SSS");
        DATE_FORMAT.put(DateFormat.ordinal_date, "yyyy-DDD");
        DATE_FORMAT.put(DateFormat.ordinal_date_time, "yyyy-DDD'T'HH:mm:ss.SSSZZ");
        DATE_FORMAT.put(DateFormat.ordinal_date_time_no_millis, "yyyy-DDD'T'HH:mm:ssZZ");
        DATE_FORMAT.put(DateFormat.time, "HH:mm:ss.SSSZZ");
        DATE_FORMAT.put(DateFormat.time_no_millis, "HH:mm:ssZZ");
        DATE_FORMAT.put(DateFormat.t_time, "'T'HH:mm:ss.SSSZZ");
        DATE_FORMAT.put(DateFormat.t_time_no_millis, "'T'HH:mm:ssZZ");
        DATE_FORMAT.put(DateFormat.week_date, "xxxx-'W'ww-e");
        DATE_FORMAT.put(DateFormat.week_date_time, "xxxx-'W'ww-e'T'HH:mm:ss.SSSZZ");
        DATE_FORMAT.put(DateFormat.weekDateTimeNoMillis, "xxxx-'W'ww-e'T'HH:mm:ssZZ");
        DATE_FORMAT.put(DateFormat.week_year, "xxxx");
        DATE_FORMAT.put(DateFormat.weekyearWeek, "xxxx-'W'ww");
        DATE_FORMAT.put(DateFormat.weekyearWeekDay, "xxxx-'W'ww-e");
        DATE_FORMAT.put(DateFormat.year, "yyyy");
        DATE_FORMAT.put(DateFormat.year_month, "yyyy-MM");
        DATE_FORMAT.put(DateFormat.year_month_day, "yyyy-MM-dd");
    }

    /**
     * 值转换
     *
     * @param value 待转换的值
     * @return 转换后的值
     */
    public Object convert(FieldWrapper field, Object value) {
        if (value instanceof Date || value instanceof TemporalAccessor) {
            return convertForDate(field, value);
        }

        return value;
    }

    /**
     * 日期类型的数据
     *
     * @param field 属性
     * @param value 值
     * @return 转换后的值
     */
    private Object convertForDate(FieldWrapper field, Object value) {
        var f = field.getElasticsearchField();
        var dateFormat = f.format();
        if (dateFormat == DateFormat.none) {
            return value;
        }

        // 自定义的日期格式
        if (dateFormat == DateFormat.custom) {
            Assert.notBlank(f.pattern(), "{}的@Field注解中pattern为空", field.getFullName());
            return formatDate(value, f.pattern());
        }

        // elasticsearch内置的日期格式
        var pattern = DATE_FORMAT.get(dateFormat);
        if (pattern != null) {
            return formatDate(value, pattern);
        }

        logger.warn("不支持转换日期格式的字段：{}.{}", field.getClazz().getName(), field.getFullName());

        return value;
    }

    private Object formatDate(Object value, String formatter) {
        return value instanceof Date ? new SimpleDateFormat(formatter).format((Date) value) : DateTimeFormatter.ofPattern(formatter).format((TemporalAccessor) value);
    }

}
