package com.elitesland.boot.elasticsearch.config;

import cn.hutool.core.util.ArrayUtil;
import cn.hutool.core.util.StrUtil;
import com.elitesland.boot.elasticsearch.ElasticsearchClientService;
import com.elitesland.boot.elasticsearch.annotation.DocumentService;
import com.elitesland.boot.elasticsearch.support.ClientServiceFactory;
import com.elitesland.boot.util.ClassPathBeanScanner;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanCreationException;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.beans.factory.config.RuntimeBeanReference;
import org.springframework.beans.factory.support.*;
import org.springframework.context.annotation.Role;
import org.springframework.lang.NonNull;
import org.springframework.util.ClassUtils;
import org.springframework.util.StopWatch;

import java.beans.Introspector;
import java.lang.reflect.ParameterizedType;
import java.util.Arrays;
import java.util.Set;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2021/6/30
 */
@Slf4j
@Role(BeanDefinition.ROLE_INFRASTRUCTURE)
public class ElasticsearchClientRegister implements BeanDefinitionRegistryPostProcessor {

    private final String[] basePackages;

    public ElasticsearchClientRegister(String[] basePackages) {
        this.basePackages = basePackages;
    }

    @Override
    public void postProcessBeanDefinitionRegistry(@NonNull BeanDefinitionRegistry registry) throws BeansException {
        // 不需要依赖其它bean时可在这里注册
    }

    @Override
    public void postProcessBeanFactory(@NonNull ConfigurableListableBeanFactory beanFactory) throws BeansException {
        if (ArrayUtil.isEmpty(basePackages)) {
            logger.warn("注册 Elasticsearch Client Service 失败：{}", Arrays.toString(basePackages));
            return;
        }

        logger.info("start register Elasticsearch Client Service，scan packages：{}", Arrays.toString(basePackages));
        var stopWatch = new StopWatch();
        stopWatch.start();

        Set<Class<?>> classSet = null;
        try {
            classSet = scan(basePackages);
        } catch (Exception exception) {
            throw new BeanCreationException("加载 Elasticsearch Client Service 失败：", exception);
        }

        // 对扫描到的接口进行动态代理注册
        registry(beanFactory, classSet);

        stopWatch.stop();
        logger.info("finish register {} Elasticsearch Client Service in {}ms", classSet.size(), stopWatch.getTotalTimeMillis());
    }

    private Set<Class<?>> scan(String[] basePackage) {
        var clazzName = ElasticsearchClientService.class.getName();

        return new ClassPathBeanScanner(basePackage)
                .filter(metadataReader -> metadataReader.getClassMetadata().isInterface()
                        && ArrayUtil.contains(metadataReader.getClassMetadata().getInterfaceNames(), clazzName))
                .scan(t -> {
                    try {
                        return Class.forName(t.getClassMetadata().getClassName());
                    } catch (ClassNotFoundException e) {
                        throw new IllegalArgumentException(e);
                    }
                });
    }

    private void registry(ConfigurableListableBeanFactory configurableListableBeanFactory, Set<Class<?>> classSet) {
        DefaultListableBeanFactory beanFactory = (DefaultListableBeanFactory) configurableListableBeanFactory;
        GenericBeanDefinition beanDefinition;
        for (Class<?> clazz : classSet) {
            beanDefinition = (GenericBeanDefinition) BeanDefinitionBuilder.genericBeanDefinition(clazz).getRawBeanDefinition();

            beanDefinition.setBeanClass(ClientServiceFactory.class);
            beanDefinition.getConstructorArgumentValues().addGenericArgumentValue(clazz);
            beanDefinition.getConstructorArgumentValues().addGenericArgumentValue(getDocType(clazz));
            beanDefinition.getPropertyValues().add("restTemplate", new RuntimeBeanReference("elasticsearchTemplate"));
            beanDefinition.setAutowireMode(AbstractBeanDefinition.AUTOWIRE_BY_TYPE);

            var beanName = generateBeanName(clazz);
            logger.info("registered Elasticsearch Client Service：{}", beanName);

            beanFactory.registerBeanDefinition(beanName, beanDefinition);
        }
    }

    private Class<?> getDocType(Class<?> clazz) {
        try {
            return (Class<?>) ((ParameterizedType) clazz.getGenericInterfaces()[0]).getActualTypeArguments()[0];
        } catch (Exception exception) {
            throw new IllegalArgumentException("注册Elasticsearch Client Service失败：" + clazz.getName());
        }
    }

    private String generateBeanName(Class<?> clazz) {
        // 先从注解中取
        var docServices = clazz.getAnnotationsByType(DocumentService.class);
        if (ArrayUtil.isNotEmpty(docServices)) {
            var beanName = docServices[0].value();
            if (StrUtil.isNotBlank(beanName)) {
                return beanName;
            }
        }

        String shortName = ClassUtils.getShortName(clazz);
        return Introspector.decapitalize(shortName);
    }
}
