package com.elitesland.boot.elasticsearch.service;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitesland.boot.elasticsearch.ElasticsearchClientService;
import com.elitesland.boot.elasticsearch.annotation.DocumentSupport;
import com.elitesland.boot.elasticsearch.common.BaseDocument;
import com.elitesland.boot.elasticsearch.common.PageResult;
import com.elitesland.boot.elasticsearch.common.query.AbstractDocumentParam;
import com.elitesland.boot.elasticsearch.common.query.AggregationParam;
import com.elitesland.boot.elasticsearch.common.query.QueryParam;
import com.elitesland.boot.elasticsearch.common.query.SearchParam;
import com.elitesland.boot.elasticsearch.service.handler.DocumentSaveHandler;
import com.elitesland.boot.elasticsearch.support.IndexSupportHelper;
import lombok.extern.slf4j.Slf4j;
import org.elasticsearch.search.aggregations.Aggregations;
import org.elasticsearch.search.sort.FieldSortBuilder;
import org.elasticsearch.search.sort.SortBuilders;
import org.elasticsearch.search.suggest.Suggest;
import org.elasticsearch.search.suggest.SuggestBuilder;
import org.elasticsearch.search.suggest.SuggestBuilders;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.elasticsearch.core.ElasticsearchRestTemplate;
import org.springframework.data.elasticsearch.core.IndexOperations;
import org.springframework.data.elasticsearch.core.SearchHit;
import org.springframework.data.elasticsearch.core.SearchHits;
import org.springframework.data.elasticsearch.core.mapping.IndexCoordinates;
import org.springframework.data.elasticsearch.core.query.NativeSearchQueryBuilder;
import org.springframework.data.repository.NoRepositoryBean;
import org.springframework.util.Assert;

import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2021/6/24
 */
@Slf4j
@NoRepositoryBean
public class ElasticsearchClientServiceSimpleImpl<T extends BaseDocument, E> implements ElasticsearchClientService<T, E> {

    private final Class<T> docType;
    private final DocumentSupport documentSupport;
    private final ElasticsearchRestTemplate restTemplate;
    private final IndexOperations indexOperations;
    private final IndexCoordinates indexCoordinates;

    private static final String FIELD_COMPLETION = "completion";

    public ElasticsearchClientServiceSimpleImpl(Class<T> docType, DocumentSupport documentSupport, ElasticsearchRestTemplate restTemplate) {
        this.docType = docType;
        this.documentSupport = documentSupport;
        this.restTemplate = restTemplate;
        indexOperations = restTemplate.indexOps(docType);
        indexCoordinates = restTemplate.getIndexCoordinatesFor(docType);
    }

    @Override
    public String getIndexName() {
        return indexCoordinates.getIndexName();
    }

    @Override
    public boolean indexExists() {
        return indexOperations.exists();
    }

    @Override
    public boolean indexCreate() {
        var exists = indexOperations.exists();
        if (exists) {
            // 如果存在，直接返回
            return true;
        }

        // 不存在则创建索引和mapping
        exists = indexOperations.create();
        if (exists) {
            new IndexSupportHelper(indexOperations, documentSupport).updateMapping();
        }
        return exists;
    }

    @Override
    public boolean indexUpdate() {
        var exists = indexOperations.exists();
        if (!exists) {
            exists = indexOperations.create();
        }

        if (exists) {
            new IndexSupportHelper(indexOperations, documentSupport).updateMapping();
        }
        return exists;
    }

    @Override
    public boolean indexDelete() {
        if (!indexOperations.exists()) {
            return true;
        }
        return indexOperations.delete();
    }

    @Override
    public T documentSave(T document) {
        if (document == null) {
            throw new IllegalArgumentException("文档不能为空");
        }

        // 保存前的预处理
        try {
            document = new DocumentSaveHandler<>(docType).beforeSave(document);
        } catch (Exception exception) {
            throw new IllegalArgumentException("Elasticsearch数据校验失败", exception);
        }

        return restTemplate.save(document);
    }

    @Override
    public List<T> documentSave(List<T> documentList) {
        if (CollUtil.isEmpty(documentList)) {
            return Collections.emptyList();
        }

        documentList = documentList.parallelStream().map(t -> {
            try {
                return new DocumentSaveHandler<>(docType).beforeSave(t);
            } catch (Exception exception) {
                throw new IllegalArgumentException("Elasticsearch数据校验失败", exception);
            }
        }).collect(Collectors.toList());

        restTemplate.save(documentList);
        return documentList;
    }

    @Override
    public boolean documentExists(E id) {
        if (id == null) {
            throw new IllegalArgumentException("文档标识为空");
        }

        return restTemplate.exists(id.toString(), indexCoordinates);
    }

    @Override
    public T documentGet(E id) {
        if (id == null) {
            throw new IllegalArgumentException("文档标识为空");
        }

        return restTemplate.get(id.toString(), docType, indexCoordinates);
    }

    @Override
    public List<T> documentGet(List<E> idList) {
        if (CollUtil.isEmpty(idList)) {
            return Collections.emptyList();
        }

        var ids = idList.stream().map(Objects::toString).distinct().collect(Collectors.toList());
        NativeSearchQueryBuilder queryBuilder = new NativeSearchQueryBuilder().withIds(ids);

        return restTemplate.multiGet(queryBuilder.build(), docType, indexCoordinates);
    }

    @Override
    public boolean documentDelete(E id) {
        if (id == null) {
            throw new IllegalArgumentException("文档标识为空");
        }

        restTemplate.delete(id.toString(), indexCoordinates);
        return true;
    }

    @Override
    public boolean documentDelete(List<E> idList) {
        if (CollUtil.isEmpty(idList)) {
            return true;
        }

        var ids = idList.stream().map(Objects::toString).distinct().collect(Collectors.toList());
        NativeSearchQueryBuilder queryBuilder = new NativeSearchQueryBuilder().withIds(ids);
        restTemplate.delete(queryBuilder.build(), docType, indexCoordinates);
        return true;
    }

    @Override
    public PageResult<T> documentQuery(QueryParam<T> queryParam) {
        var queryBuilder = new NativeSearchQueryBuilder();

        // 条件
        queryBuilder.withFilter(queryParam.getFilterBuilder() == null ? null : queryParam.getFilterBuilder().build());

        // 排序
        if (queryParam.getOrderBuilder() != null) {
            for (FieldSortBuilder b : queryParam.getOrderBuilder().build()) {
                queryBuilder.withSort(b);
            }
        }

        // 分页
        queryBuilder.withPageable(obtainPageRequest(queryParam));

        SearchHits<T> searchHits = restTemplate.search(queryBuilder.build(), docType, indexCoordinates);
        List<T> records = searchHits.hasSearchHits() ? searchHits.getSearchHits().stream()
                .map(SearchHit::getContent)
                .collect(Collectors.toList()) : Collections.emptyList();
        return new PageResult<>(searchHits.getTotalHits(), records);
    }

    @Override
    public PageResult<T> documentSearch(SearchParam<T> searchParam) {
        var queryBuilder = new NativeSearchQueryBuilder();

        // 条件
        queryBuilder.withFilter(searchParam.getFilterBuilder() == null ? null : searchParam.getFilterBuilder().build());
        queryBuilder.withQuery(searchParam.getQueryBuilder() == null ? null : searchParam.getQueryBuilder().build());

        // 排序
        queryBuilder.withSort(SortBuilders.scoreSort());

        // 分页
        queryBuilder.withPageable(obtainPageRequest(searchParam));

        SearchHits<T> searchHits = restTemplate.search(queryBuilder.build(), docType, indexCoordinates);
        List<T> records = searchHits.hasSearchHits() ? searchHits.getSearchHits().stream()
                .map(SearchHit::getContent)
                .collect(Collectors.toList()) : Collections.emptyList();
        return new PageResult<>(searchHits.getTotalHits(), records);
    }

    @Override
    public Aggregations aggregationSearch(AggregationParam<T> aggregationParam) {
        var queryBuilder = new NativeSearchQueryBuilder();

        // 条件
        if (aggregationParam.getFilterBuilder() != null) {
            queryBuilder.withFilter(aggregationParam.getFilterBuilder().build());
        }

        // 聚合条件
        Assert.notEmpty(aggregationParam.getAggregationBuilders(), "请配置聚合参数");
        for (var agg : aggregationParam.getAggregationBuilders()) {
            queryBuilder.addAggregation(agg);
        }

        queryBuilder.withPageable(PageRequest.of(0, 1));

        var searchHits = restTemplate.search(queryBuilder.build(), docType, indexCoordinates);
        return searchHits.getAggregations();
    }

    @Override
    public List<String> suggestComplete(String prefix, Integer size) {
        if (CharSequenceUtil.isBlank(prefix)) {
            return Collections.emptyList();
        }
        if (size == null || size < 1) {
            size = 10;
        }

        var completionSuggestBuilder = SuggestBuilders.completionSuggestion(FIELD_COMPLETION)
                .prefix(prefix).skipDuplicates(true).size(size);
        var suggestBuilder = new SuggestBuilder();
        suggestBuilder.addSuggestion(Suggest.NAME, completionSuggestBuilder);

        return restTemplate.suggest(suggestBuilder, indexCoordinates)
                .getSuggest()
                .getSuggestion(Suggest.NAME)
                .getEntries()
                .stream()
                .flatMap(t -> t.getOptions().stream().map(tt -> tt.getText().toString()))
                .collect(Collectors.toList());
    }

    private PageRequest obtainPageRequest(AbstractDocumentParam<T> param) {
        int page = Math.max(1, param.getPage()) - 1;
        int pageSize = Math.max(1, param.getPageSize());
        pageSize = Math.min(pageSize, 1000);

        return PageRequest.of(page, pageSize);
    }
}
