package com.elitesland.boot.elasticsearch.service.handler;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.StrUtil;
import com.elitesland.boot.elasticsearch.common.BaseDocument;
import com.elitesland.boot.elasticsearch.support.DocumentIndexCache;
import com.elitesland.boot.elasticsearch.support.FieldWrapper;
import com.elitesland.boot.util.ObjectResolver;
import org.springframework.data.annotation.Transient;
import org.springframework.data.elasticsearch.core.completion.Completion;

import java.lang.reflect.Field;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 保存前的处理.
 *
 * @author Kaiser（wang shao）
 * @date 2021/07/15
 */
public class DocumentSaveHandler<T extends BaseDocument> {

    /**
     * 文档类型
     */
    private final Class<T> documentType;
    /**
     * 文档索引配置
     */
    private final DocumentIndexCache.DocumentIndex documentIndex;

    public DocumentSaveHandler(Class<T> documentType) {
        this.documentType = documentType;
        this.documentIndex = DocumentIndexCache.getDocumentIndex(documentType);
        Assert.notNull(documentIndex, "索引数据失败，初始化异常");
    }

    /**
     * 前置处理器
     *
     * @param document 待保存的文档
     * @return 文档
     */
    public T beforeSave(T document) throws Exception {
        Assert.notNull(document.getId(), "文档的唯一标识不能为空");

        var needFilter = documentIndex.isNeedFilter();
        if (!needFilter) {
            return document;
        }

        Set<String> completionValues = new HashSet<>(64);
        for (var field : documentIndex.getFieldWrapperMap().values()) {
            // 获取需要自动补全的字段
            obtainCompletionValue(document, field, completionValues);

            // 去掉需要忽略的值
            eraseValue(document, field);
        }

        writeCompletionValue(document, completionValues);

        if (document.getCreateTime() == null) {
            document.setCreateTime(LocalDateTime.now());
        }

        return document;
    }

    private void writeCompletionValue(T document, Set<String> completionValues) {
        if (completionValues.isEmpty()) {
            return;
        }
        document.setCompletion(new Completion(new ArrayList<>(completionValues)));
    }

    private void eraseValue(T document, FieldWrapper fieldWrapper) throws Exception {
        if (!fieldWrapper.isIgnore()) {
            return;
        }
        if (fieldWrapper.isPrimitive()) {
            Assert.isTrue(isTransient(fieldWrapper.getField()), "{}请使用包装类型", fieldWrapper.getField().toGenericString());
        }

        erase(document, null, fieldWrapper);
    }

    private void erase(Object source, String fieldPrefix, FieldWrapper fieldWrapper) throws Exception {
        if (StrUtil.equals(fieldPrefix, fieldWrapper.getNamePrefix())) {
            // 将需要忽略的字段设置成null
            fieldWrapper.getPropertyDescriptor().getWriteMethod().invoke(source, new Object[]{null});
            return;
        }

        var propertyName = obtainPropertyName(fieldWrapper.getFullName(), fieldPrefix);
        var fieldName = (fieldPrefix == null ? "" : (fieldPrefix + ".")) + propertyName;
        Object value = documentIndex.getFieldWrapperMap().get(fieldName).getPropertyDescriptor().getReadMethod().invoke(source);

        ObjectResolver.resolve(value, t -> {
            try {
                erase(t, fieldName, fieldWrapper);
            } catch (Exception exception) {
                throw new IllegalArgumentException(exception);
            }
        });
    }

    private void obtainCompletionValue(T document, FieldWrapper fieldWrapper, Set<String> completionValues) throws Exception {
        if (!fieldWrapper.isCompletionSource()) {
            return;
        }

        completionValue(document, null, fieldWrapper, completionValues);
    }

    private void completionValue(Object source, String fieldPrefix, FieldWrapper fieldWrapper, Set<String> completionValues) throws Exception {
        if (StrUtil.equals(fieldPrefix, fieldWrapper.getNamePrefix())) {
            Object value = fieldWrapper.getPropertyDescriptor().getReadMethod().invoke(source);
            ObjectResolver.resolve(value, t -> completionValues.add((String) t));
            return;
        }

        var propertyName = obtainPropertyName(fieldWrapper.getFullName(), fieldPrefix);
        var fieldName = (fieldPrefix == null ? "" : (fieldPrefix + ".")) + propertyName;
        Object value = documentIndex.getFieldWrapperMap().get(fieldName).getPropertyDescriptor().getReadMethod().invoke(source);
        ObjectResolver.resolve(value, t -> {
            try {
                completionValue(t, fieldPrefix, fieldWrapper, completionValues);
            } catch (Exception exception) {
                throw new IllegalArgumentException(exception);
            }
        });
    }

    private boolean isTransient(Field field) {
        return field.isAnnotationPresent(Transient.class);
    }

    private static String obtainPropertyName(String str, String prefix) {
        if (prefix == null) {
            return str.substring(0, str.indexOf("."));
        }
        var offset = prefix.length() + 1;
        var index = str.indexOf(".", offset);
        return str.substring(offset, index == -1 ? str.length() : index);
    }
}
