package com.elitesland.boot.elasticsearch.support;

import cn.hutool.core.util.ArrayUtil;
import com.elitesland.boot.elasticsearch.annotation.DocumentSupport;
import com.elitesland.boot.elasticsearch.common.BaseDocument;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.cglib.core.ReflectUtils;
import org.springframework.data.elasticsearch.core.ElasticsearchRestTemplate;

import java.beans.PropertyDescriptor;
import java.lang.reflect.Proxy;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 服务实例工厂.
 *
 * @author Kaiser（wang shao）
 * @date 2021/6/26
 */
public class ClientServiceFactory<B, T extends BaseDocument, E> implements FactoryBean<B> {

    /**
     * 被代理的接口
     */
    private final Class<B> target;
    /**
     * 索引的文档类型
     */
    private final Class<T> docType;
    private final DocumentSupport documentSupport;
    private ElasticsearchRestTemplate restTemplate;

    public ClientServiceFactory(Class<B> target, Class<T> docType) {
        this.target = target;
        this.docType = docType;

        var documentSupports = docType.getAnnotationsByType(DocumentSupport.class);
        this.documentSupport = ArrayUtil.isEmpty(documentSupports) ? null : documentSupports[0];

        // 参数校验
        validate(docType, documentSupport);
    }

    @SuppressWarnings("unchecked")
    @Override
    public B getObject() throws Exception {
        return (B) Proxy.newProxyInstance(target.getClassLoader(), new Class[]{target},
                new ClientServiceProxy<T, E>(docType, documentSupport, restTemplate));
    }

    @Override
    public Class<?> getObjectType() {
        return target;
    }

    @Override
    public boolean isSingleton() {
        return true;
    }

    public void setRestTemplate(ElasticsearchRestTemplate restTemplate) {
        this.restTemplate = restTemplate;
    }

    private void validate(Class<T> docType, DocumentSupport documentSupport) {
        if (documentSupport == null) {
            throw new IllegalArgumentException("在【" + docType.getName() + "】上未找到@DocumentSupport()的注解！");
        }

        validateCompletion(docType, documentSupport);
    }

    private void validateCompletion(Class<T> docType, DocumentSupport documentSupport) {
        // 校验自动补全字段
        var fields = documentSupport.completionFields();
        if (ArrayUtil.isEmpty(fields)) {
            return;
        }

        var properties = ReflectUtils.getBeanProperties(docType);
        var validated = true;
        if (ArrayUtil.isEmpty(properties)) {
            validated = false;
        } else {
            var propertiesMap = Arrays.stream(properties).collect(Collectors.toMap(PropertyDescriptor::getName, t -> t, (t1, t2) -> t1));
            Set<String> existsField = new HashSet<>(fields.length);
            for (var f : fields) {
                var property = propertiesMap.get(f);
                if (property == null || property.getReadMethod() == null || property.getWriteMethod() == null || existsField.contains(f)) {
                    validated = false;
                    break;
                }
                existsField.add(f);
            }
        }
        if (!validated) {
            throw new IllegalArgumentException("在【" + docType.getName() + "】中配置有误，请检查");
        }
    }

}
