package com.elitesland.boot.elasticsearch.support;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ArrayUtil;
import cn.hutool.core.util.ReflectUtil;
import com.elitesland.boot.elasticsearch.annotation.DocumentSupport;
import com.elitesland.boot.elasticsearch.common.BaseDocument;
import lombok.Data;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.springframework.cglib.core.ReflectUtils;
import org.springframework.data.elasticsearch.annotations.Field;
import org.springframework.data.elasticsearch.annotations.FieldType;
import org.springframework.data.elasticsearch.core.completion.Completion;
import org.springframework.data.mapping.model.SimpleTypeHolder;
import org.springframework.util.StopWatch;

import java.lang.reflect.ParameterizedType;
import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 文档索引.
 *
 * @author Kaiser（wang shao）
 * @date 2021/07/09
 */
@Slf4j
public class DocumentIndexCache {

    private static final Map<String, DocumentIndex> CACHE = new HashMap<>(64);
    private static SimpleTypeHolder simpleTypeHolder = null;
    /**
     * 忽略解析的字段类型
     */
    private static final Set<Class<?>> IGNORE_FIELD_TYPE = new HashSet<>(64);

    private DocumentIndexCache() {
    }

    /**
     * 初始化缓存
     *
     * @param classSet 文档model类
     */
    public static void init(Set<Class<?>> classSet) {
        if (CollUtil.isEmpty(classSet)) {
            return;
        }

        var stopWatch = new StopWatch();
        stopWatch.start();

        Class<?> clazzBase = BaseDocument.class;
        for (Class<?> clazz : classSet) {
            if (!clazzBase.isAssignableFrom(clazz)) {
                continue;
            }
            CACHE.put(clazz.getName(), buildDocumentIndex(clazz));
        }
        stopWatch.stop();
        logger.info("Elasticsearch document index config cache loaded finished for {} in {} ms", classSet.size(), stopWatch.getTotalTimeMillis());
    }

    /**
     * 获取索引配置
     *
     * @param clazz 文档model类
     * @return 配置
     */
    public static DocumentIndex getDocumentIndex(Class<?> clazz) {
        return CACHE.get(clazz.getName());
    }

    static {
        Set<Class<?>> customSimpleSet = new HashSet<>(64);
        customSimpleSet.add(BigDecimal.class);
        customSimpleSet.add(LocalDateTime.class);
        customSimpleSet.add(LocalDate.class);
        customSimpleSet.add(LocalTime.class);
        simpleTypeHolder = new SimpleTypeHolder(customSimpleSet, true);

        IGNORE_FIELD_TYPE.add(Completion.class);
    }

    private static DocumentIndex buildDocumentIndex(Class<?> clazz) {
        var documentIndex = new DocumentIndex();

        // 文档注解
        documentIndex.setDocumentSupport(obtainDocumentSupport(clazz));
        documentIndex.setFields(new HashSet<>(64));
        documentIndex.setFieldWrapperMap(new HashMap<>(64));

        // 分析属性
        analyseObjectProperty(documentIndex, clazz, null);

        // 判断是否需要过滤数据
        setNeedFilter(documentIndex);

        return documentIndex;
    }

    private static void setNeedFilter(DocumentIndex documentIndex) {
        if (ArrayUtil.isNotEmpty(documentIndex.getDocumentSupport().completionFields())) {
            // 需要处理自动补全字段
            documentIndex.setNeedFilter(true);
            return;
        }

        var needIgnore = documentIndex.getFieldWrapperMap().values().stream().anyMatch(FieldWrapper::isIgnore);
        documentIndex.setNeedFilter(needIgnore);
    }

    private static void verifyAnnotation(java.lang.reflect.Field field, Field annotation) {
        if (annotation == null) {
            return;
        }

        var validate = false;
        if (isSimpleType(field.getType())) {
            // 简单类型时
            validate = annotation.type() == FieldType.Object || annotation.type() == FieldType.Nested;
            Assert.isFalse(validate, "{}不支持使用注解：{}", field.toGenericString(), annotation.type());
        }
    }

    private static void analyseObject(DocumentIndex documentIndex, java.lang.reflect.Field field, Field annotation, String fieldPrefix) {
        verifyAnnotation(field, annotation);

        // 获取数据类型
        var type = field.getType();
        if (type.isArray()) {
            // 数组时
            type = type.getComponentType();
        } else if (Collection.class.isAssignableFrom(type)) {
            var types = ((ParameterizedType) field.getGenericType()).getActualTypeArguments();
            Assert.isFalse(ArrayUtil.isEmpty(types), "{}的类型不明确", field.toGenericString());
            type = (Class<?>) types[0];
        }
        if (isSimpleType(type)) {
            // 简单类型，则直接返回
            return;
        }

        analyseObjectProperty(documentIndex, type, fieldPrefix);
    }

    private static void analyseObjectProperty(DocumentIndex documentIndex, Class<?> clazz, String fieldPrefix) {
        if (IGNORE_FIELD_TYPE.contains(clazz)) {
            return;
        }
        // 所有的属性
        var properties = ReflectUtils.getBeanProperties(clazz);
        var fields = ReflectUtil.getFields(clazz);
        if (ArrayUtil.isEmpty(properties) || ArrayUtil.isEmpty(fields)) {
            return;
        }
        var fieldMap = Arrays.stream(fields).collect(Collectors.toMap(java.lang.reflect.Field::getName, t -> t, (t1, t2) -> t1));

        FieldWrapper fieldWrapper = null;
        for (var p : properties) {
            fieldWrapper = new FieldWrapper(documentIndex.getDocumentSupport(), clazz, fieldPrefix,
                    fieldMap.get(p.getName()), p, simpleTypeHolder);

            documentIndex.getFieldWrapperMap().put(fieldWrapper.getFullName(), fieldWrapper);

            // 判断类型，如果不是简单类型，则处理子类型
            analyseObject(documentIndex, fieldWrapper.getField(), fieldWrapper.getElasticsearchField(), fieldWrapper.getFullName());

            if (!fieldWrapper.isIgnore()) {
                documentIndex.getFields().add(fieldWrapper.getFullName());
            }
        }
    }

    private static DocumentSupport obtainDocumentSupport(Class<?> clazz) {
        var annotations = clazz.getAnnotationsByType(DocumentSupport.class);
        if (ArrayUtil.isEmpty(annotations)) {
            return null;
        }
        return annotations[0];
    }

    /**
     * 是否是简单类型
     *
     * @param type 类型
     * @return 是否
     */
    private static boolean isSimpleType(Class<?> type) {
        return simpleTypeHolder.isSimpleType(type);
    }

    @Getter
    public static class DocumentIndex {
        /**
         * 文档上的注解
         */
        private DocumentSupport documentSupport;

        /**
         * 所有与索引中对应的属性
         */
        private Set<String> fields = Collections.emptySet();

        /**
         * 属性描述
         */
        private Map<String, FieldWrapper> fieldWrapperMap = Collections.emptyMap();

        /**
         * 是否需要对数据进行过滤处理
         */
        private boolean needFilter;

        void setDocumentSupport(DocumentSupport documentSupport) {
            this.documentSupport = documentSupport;
        }

        void setFields(Set<String> fields) {
            this.fields = fields;
        }

        void setFieldWrapperMap(Map<String, FieldWrapper> fieldWrapperMap) {
            this.fieldWrapperMap = fieldWrapperMap;
        }

        void setNeedFilter(boolean needFilter) {
            this.needFilter = needFilter;
        }
    }
}
