package com.elitesland.boot.elasticsearch.support;

import cn.hutool.core.util.ArrayUtil;
import com.elitesland.boot.elasticsearch.annotation.DocumentSupport;
import com.elitesland.boot.util.ClassPathBeanScanner;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.elasticsearch.annotations.Document;
import org.springframework.data.elasticsearch.core.ElasticsearchRestTemplate;
import org.springframework.util.StopWatch;

import java.util.Arrays;
import java.util.Collections;
import java.util.Set;

/**
 * 索引检查.
 *
 * @author Kaiser（wang shao）
 * @date 2021/7/2
 */
@Slf4j
class DocumentIndexCheck {

    private final ElasticsearchRestTemplate template;
    private final String[] basePackages;

    private static final Class<DocumentSupport> ANNOTATION = DocumentSupport.class;

    public DocumentIndexCheck(ElasticsearchRestTemplate template, String[] basePackages) {
        this.template = template;
        this.basePackages = basePackages;
    }

    public Set<Class<?>> check() {
        if (ArrayUtil.isEmpty(basePackages) || template == null) {
            logger.warn("Elasticsearch Document Index check fail：{}，{}", template == null, basePackages);
            return Collections.emptySet();
        }

        logger.info("start scan Elasticsearch document package：{}", Arrays.toString(basePackages));
        var stopWatch = new StopWatch();
        stopWatch.start();

        // 扫描获取所有的文档索引
        Set<Class<?>> entitySet = null;
        try {
            entitySet = scanPackageByAnnotation(basePackages);
        } catch (Exception e) {
            throw new IllegalArgumentException("Elasticsearch document实体初始化失败", e);
        }

        // 检查索引
        for (Class<?> cls : entitySet) {
            checkIndex(cls);
        }

        stopWatch.stop();
        logger.info("finish scan Elasticsearch document in {}ms", stopWatch.getTotalTimeMillis());

        return entitySet;
    }

    private void checkIndex(Class<?> clazz) {
        DocumentSupport[] annotations = clazz.getAnnotationsByType(ANNOTATION);
        if (ArrayUtil.isEmpty(annotations)) {
            throw new RuntimeException("索引Elasticsearch索引检查失败");
        }

        DocumentSupport annotation = annotations[0];
        switch (annotation.checkStrategy()) {
            case NO_CHECK:
                noCheck(clazz);
                break;
            case CREATE_IF_NO_EXIST:
                createIfNotExist(clazz, annotation);
                break;
            case ALWAYS_UPDATE:
                alwaysUpdate(clazz, annotation);
                break;
            default:
                break;
        }
    }

    private Set<Class<?>> scanPackageByAnnotation(String[] packageNames) throws RuntimeException {
        var annotationName = ANNOTATION.getName();
        var documentAnnotationName = Document.class.getName();
        return new ClassPathBeanScanner(packageNames)
                .filter(metadataReader -> metadataReader.getAnnotationMetadata().hasAnnotation(annotationName) &&
                        metadataReader.getAnnotationMetadata().hasAnnotation(documentAnnotationName)
                )
                .scan(t -> {
                    try {
                        return Class.forName(t.getClassMetadata().getClassName());
                    } catch (ClassNotFoundException e) {
                        throw new IllegalArgumentException(e);
                    }
                });
    }

    private void noCheck(Class<?> index) {
        var exists = false;
        try {
            var indexOperations = template.indexOps(index);
            exists = indexOperations.exists();
        } catch (Exception e) {
            logger.error("检查索引【{}】是否存在失败：{}", index.getName(), e.getMessage());
            return;
        }
        logger.info("index【{}】exists：{}", index.getName(), exists);
    }

    private void createIfNotExist(Class<?> index, DocumentSupport documentSupport) {
        try {
            var indexOperations = template.indexOps(index);
            if (indexOperations.exists()) {
                // 存在则直接返回
                logger.info("index【{}】 has exists", index.getName());
                return;
            }

            var created = indexOperations.create();
            if (created) {
                new IndexSupportHelper(indexOperations, documentSupport).updateMapping();
            }
            logger.info("index【{}】 has updated：{}", index.getName(), created);
        } catch (Exception e) {
            logger.error("检查索引【{}】是否存在失败：{}", index.getName(), e.getMessage());
        }
    }

    private void alwaysUpdate(Class<?> index, DocumentSupport documentSupport) {
        try {
            var indexOperations = template.indexOps(index);

            var exists = indexOperations.exists();
            if (!exists) {
                // 不存在则先创建
                exists = indexOperations.create();
            }

            if (exists) {
                new IndexSupportHelper(indexOperations, documentSupport).updateMapping();
            }
            logger.info("index【{}】 has updated：{}", index.getName(), exists);
        } catch (Exception e) {
            logger.error("检查索引【{}】是否存在失败：{}", index.getName(), e.getMessage());
        }
    }

}
