package com.elitesland.boot.elasticsearch.support;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ArrayUtil;
import cn.hutool.core.util.StrUtil;
import com.elitesland.boot.elasticsearch.annotation.DocumentSupport;
import lombok.Data;
import org.springframework.data.elasticsearch.annotations.Field;
import org.springframework.data.mapping.model.SimpleTypeHolder;

import java.beans.PropertyDescriptor;
import java.lang.reflect.ParameterizedType;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 属性.
 *
 * @author Kaiser（wang shao）
 * @date 2021/07/15
 */
@Data
public class FieldWrapper {

    /**
     * 文档注解
     */
    private DocumentSupport documentSupport;
    /**
     * 所在类
     */
    private Class<?> clazz;
    /**
     * 属性名称前缀
     */
    private String namePrefix;
    /**
     * 属性
     */
    private java.lang.reflect.Field field;
    /**
     * 属性描述
     */
    private PropertyDescriptor propertyDescriptor;

    private SimpleTypeHolder simpleTypeHolder;

    /**
     * 属性名称
     */
    private String name;
    /**
     * 完整名称
     */
    private String fullName;

    /**
     * 实际的参数类型
     */
    private Class<?> actualType;

    /**
     * 字段注解
     */
    private Field elasticsearchField;

    /**
     * 是否忽略写入Elasticsearch
     */
    private boolean ignore;

    /**
     * 是否是自动补全字段数据来源
     */
    private boolean completionSource;

    /**
     * 是否是基本数据类型
     */
    private boolean primitive;

    /**
     * 是否是简单数据类型
     */
    private boolean simple;

    /**
     * 是否是数组
     */
    private boolean array;

    /**
     * 是否是集合
     */
    private boolean collection;

    public FieldWrapper(DocumentSupport documentSupport, Class<?> clazz, String namePrefix, java.lang.reflect.Field field,
                        PropertyDescriptor propertyDescriptor, SimpleTypeHolder simpleTypeHolder) {
        this.documentSupport = documentSupport;
        this.clazz = clazz;
        this.namePrefix = namePrefix;
        this.field = field;
        this.propertyDescriptor = propertyDescriptor;
        this.simpleTypeHolder = simpleTypeHolder;

        init();
    }

    private void init() {
        this.name = propertyDescriptor.getName();
        this.fullName = this.namePrefix == null ? this.name : this.namePrefix + "." + this.name;
        this.elasticsearchField = ArrayUtil.get(field.getAnnotationsByType(Field.class), 0);
        this.ignore = this.elasticsearchField == null;

        dealType();
        dealCompletion();
    }

    private void dealType() {
        var type = propertyDescriptor.getPropertyType();
        primitive = type.isPrimitive();
        simple = simpleTypeHolder.isSimpleType(type);
        array = type.isArray();
        collection = Collection.class.isAssignableFrom(type);

        if (primitive || simple) {
            this.actualType = type;
        } else {
            if (array) {
                // 数组时
                this.actualType = type.getComponentType();
            } else if (Collection.class.isAssignableFrom(type)){
                // 集合时
                var types = ((ParameterizedType) field.getGenericType()).getActualTypeArguments();
                Assert.isFalse(ArrayUtil.isEmpty(types), "{}的类型不明确", field.toGenericString());
                this.actualType = (Class<?>) types[0];
            } else {
                this.actualType = type;
            }
        }
    }

    private void dealCompletion() {
        Set<String> completionFields = ArrayUtil.isEmpty(documentSupport.completionFields()) ? Collections.emptySet()
                : Arrays.stream(documentSupport.completionFields()).collect(Collectors.toSet());
        if (completionFields.isEmpty()) {
            return;
        }
        this.completionSource = completionFields.contains(this.fullName);
        if (this.completionSource) {
            Assert.isTrue(String.class.isAssignableFrom(this.actualType), "{}字段类型不支持自动补全字段");
        }
    }

    public Class<?> getClazz() {
        return clazz;
    }

    public String getNamePrefix() {
        return namePrefix;
    }

    public java.lang.reflect.Field getField() {
        return field;
    }

    public PropertyDescriptor getPropertyDescriptor() {
        return propertyDescriptor;
    }

    public SimpleTypeHolder getSimpleTypeHolder() {
        return simpleTypeHolder;
    }

    public String getName() {
        return name;
    }

    public String getFullName() {
        return fullName;
    }

    public Field getElasticsearchField() {
        return elasticsearchField;
    }

    public boolean isIgnore() {
        return ignore;
    }

    public boolean isPrimitive() {
        return primitive;
    }

    public boolean isSimple() {
        return simple;
    }

    public boolean isArray() {
        return array;
    }

    public boolean isCollection() {
        return collection;
    }

    public Class<?> getActualType() {
        return actualType;
    }

    public boolean isCompletionSource() {
        return completionSource;
    }
}
