package com.elitesland.boot.util;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.io.support.ResourcePatternResolver;
import org.springframework.core.io.support.ResourcePatternUtils;
import org.springframework.core.type.classreading.CachingMetadataReaderFactory;
import org.springframework.core.type.classreading.MetadataReader;
import org.springframework.core.type.classreading.MetadataReaderFactory;
import org.springframework.util.ClassUtils;

import java.io.IOException;
import java.util.Arrays;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 类扫描器.
 *
 * @author Kaiser（wang shao）
 * @date 2021/6/27
 */
@Slf4j
public class ClassPathBeanScanner {

    private final String[] packageNames;

    private Function<MetadataReader, Boolean> filter;

    public ClassPathBeanScanner(String... packageNames) {
        this.packageNames = packageNames;
    }

    public ClassPathBeanScanner filter(Function<MetadataReader, Boolean> filter) {
        this.filter = filter;
        return this;
    }

    public Set<String> scan() {
        return scan(metadataReader -> metadataReader.getClassMetadata().getClassName());
    }

    public <T> Set<T> scan(@NonNull Function<MetadataReader, T> convert) {
        var pathResolver = ResourcePatternUtils.getResourcePatternResolver(null);
        MetadataReaderFactory metadataReaderFactory = new CachingMetadataReaderFactory(pathResolver);

        return Arrays.stream(packageNames).flatMap(p -> {
            try {
                return scan(pathResolver, metadataReaderFactory, convert, p).stream();
            } catch (Exception e) {
                throw new IllegalArgumentException("加载Elasticsearch Client Service 加载失败：", e);
            }
        })
                .collect(Collectors.toSet());
    }

    private <T> Set<T> scan(ResourcePatternResolver pathResolver, MetadataReaderFactory metadataReaderFactory,
                            Function<MetadataReader, T> convert, String packageName) throws IOException {
        var pattern = ResourcePatternResolver.CLASSPATH_ALL_URL_PREFIX + ClassUtils.convertClassNameToResourcePath(packageName) + "/**/*.class";

        return Arrays.stream(pathResolver.getResources(pattern)).parallel().
                map(res -> {
                    try {
                        return metadataReaderFactory.getMetadataReader(res);
                    } catch (IOException e) {
                        logger.error("ElasticsearchStarter 加载资源过程异常", e);
                        return null;
                    }
                }).filter(t -> t != null && (filter == null || filter.apply(t)))
                .map(convert)
                .collect(Collectors.toSet());
    }

}
