package com.elitesland.cbpl.sns.inbox.mongo;

import com.elitescloud.cloudt.common.base.param.OrderItem;
import com.elitesland.cbpl.sns.inbox.vo.param.InboxPageParamVO;
import lombok.Builder;
import lombok.ToString;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.util.StringUtils;

import java.util.List;

import static org.springframework.data.mongodb.core.query.Criteria.where;

/**
 * 站内信查询
 *
 * @author eric.hao
 * @since 2022/09/15
 */
@Slf4j
@Builder
@ToString
public class InboxPageQuery extends InboxMongoQuery {

    private final InboxPageParamVO params;

    @Override
    public String getStoreName() {
        return params.getType().getStoreName();
    }

    /**
     * 缺省按时间倒序排列
     *
     * @param params 查询参数(为空表示无参数)
     */
    public InboxPageQuery(InboxPageParamVO params) {
        OrderItem order = new OrderItem();
        order.setAsc(false);
        order.setColumn("_id");
        super.setOrders(List.of(order));
        super.setCurrent(params.getCurrent() + 1);
        super.setSize(params.getSize());
        this.params = params;
    }

    private void addRecipientsCriteria(Query query) {
        query.fields().exclude("tos");
        query.addCriteria(new Criteria().orOperator(
            where("tos").size(0),
            where("tos").is(params.getRecipientId())
        ));
    }

    private void addCreateTimeCriteria(Query query) {
        val sinceTime = params.getSinceTime();
        val untilTime = params.getUntilTime();
        if (sinceTime != null || untilTime != null) {
            val filter = where("time");
            if (sinceTime != null) {
                filter.gte(sinceTime);
            }
            if (untilTime != null) {
                filter.lt(untilTime);
            }
            query.addCriteria(filter);
        }
    }

    private void addReadTimesCriteria(Query query) {
        if (params.getUnread() != null) {
            query.addCriteria(where("read." + params.getRecipientId()).exists(!params.getUnread()));
        }
    }

    private void withFields(Query query) {
        if (params.isOutline()) {
            query.fields().exclude("payload.content");
        }
    }

    private void addMsgType(Query query) {
        if (params.getMsgType() != null && !params.getMsgType().equals("")) {
            query.addCriteria(where("extInfo.eventType").in(params.getMsgType()));
        }
    }

    private void addSubject(Query query) {
        val subject = params.getSubject();
        if (StringUtils.hasText(subject)) {
            val regex = "^.*" + subject + ".*$";
            query.addCriteria(Criteria.where("subject").regex(regex));
        }
    }

    @Override
    protected Query buildQuery() {
        Query query = new Query();
        addRecipientsCriteria(query);
        addCreateTimeCriteria(query);
        addReadTimesCriteria(query);
        addSubject(query);
        withFields(query);
        addMsgType(query);
        logger.info("[IM-SNS] query: {}", query);
        return query;
    }

}
