package com.elitesland.cbpl.sns.mail.provider;

import com.elitesland.cbpl.sns.mail.domain.RecipientMethod;
import com.elitesland.cbpl.sns.mail.iam.UserResolver;
import com.elitesland.cbpl.sns.mail.vo.save.MailSaveParamVO;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.io.InputStreamResource;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.mail.javamail.MimeMessageHelper;
import org.springframework.stereotype.Service;

import javax.mail.Message;
import javax.mail.internet.MimeMessage;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Calendar;

/**
 * 用于构建邮件内容
 *
 * @author eric.hao
 * @since 2023/04/17
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class MailProviderService implements MailProvider {

    private final JavaMailSender mailSender;

    private final UserResolver userInfoResolver;

    @Override
    public void send(MailSaveParamVO payload) {
        mailSender.send(createMailMessage(payload));
    }

    @SneakyThrows
    private MimeMessage createMailMessage(MailSaveParamVO payload) {
        var messageHelper = new MimeMessageHelper(mailSender.createMimeMessage(), payload.hasAttachments());
        messageHelper.setSentDate(Calendar.getInstance().getTime());
        var senderMail = userInfoResolver.resolveMail(payload.getFrom()).orElseThrow(() ->
                new IllegalArgumentException("[PHOENIX-SNS] Mail sender is unavailable: " + payload.getFrom()));
        messageHelper.setFrom(senderMail);
        messageHelper.setReplyTo(payload.getReplyTo().orElse(senderMail));
        // 直接使用邮箱发件
        if (payload.getRecipientMethod().equals(RecipientMethod.EMAIL)) {
            for (var to : payload.getTos()) {
                addMailTo(messageHelper, to);
            }
            for (var cc : payload.getCcs()) {
                addMailCc(messageHelper, cc);
            }
        }
        // 根据UserId获取邮箱
        else {
            for (var to : payload.getTos()) {
                userInfoResolver.resolveMail(to).ifPresent(i -> addMailTo(messageHelper, i));
            }
            for (var cc : payload.getCcs()) {
                userInfoResolver.resolveMail(cc).ifPresent(i -> addMailCc(messageHelper, i));
            }
        }
        messageHelper.setSubject(payload.getSubject());
        messageHelper.setText(payload.getContentText(), payload.isHtmlContent());
        for (var attachment : payload.getAttachments()) {
            addAttachment(messageHelper, attachment);
        }
        var message = messageHelper.getMimeMessage();
        if (message.getRecipients(Message.RecipientType.TO).length == 0) {
            throw new IllegalArgumentException("[PHOENIX-SNS] Mail recipients is unavailable: " + payload.getTos());
        }
        return message;
    }

    @SneakyThrows
    private void addMailTo(MimeMessageHelper messageHelper, String to) {
        messageHelper.addTo(to);
    }

    @SneakyThrows
    private void addMailCc(MimeMessageHelper messageHelper, String cc) {
        messageHelper.addCc(cc);
    }

    @SneakyThrows
    private void addAttachment(MimeMessageHelper messageHelper, Path attachment) {
        try (var attachmentStream = Files.newInputStream(attachment)) {
            messageHelper.addAttachment(attachment.getFileName().toString(), new InputStreamResource(attachmentStream));
        } catch (IOException e) {
            throw new IllegalArgumentException("[PHOENIX-SNS] Add attachment FAILED: " + attachment, e);
        }
    }

}
