package com.elitesland.cbpl.sns.mail.service;

import com.elitesland.cbpl.sns.mail.config.EmailProperties;
import com.elitesland.cbpl.sns.mail.domain.RecipientMethod;
import com.elitesland.cbpl.sns.mail.provider.MailProvider;
import com.elitesland.cbpl.sns.mail.vo.save.DefaultMailSaveParamVO;
import com.elitesland.cbpl.sns.notifier.vo.payload.SnsPayload;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.util.Set;

/**
 * @author eric.hao
 * @since 2022/09/15
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class MailServiceImpl implements MailService {

    private final EmailProperties emailProperties;
    private final MailProvider mailer;

    @Override
    public void sendByUserId(Set<String> tos, SnsPayload payload) {
        if (preCheck(payload)) {
            tos.forEach(to -> sendEmail(to, payload, RecipientMethod.USER_ID));
        }
    }

    @Override
    public void sendByEmail(Set<String> tos, SnsPayload payload) {
        if (preCheck(payload)) {
            tos.forEach(to -> sendEmail(to, payload, RecipientMethod.EMAIL));
        }
    }

    /**
     * 预检查
     *
     * @param payload 请求参数
     * @return true可以发送；false不可以发送；
     */
    private boolean preCheck(SnsPayload payload) {
        if (!emailProperties.isEnabled()) {
            logger.info("[PHOENIX-SNS] EMAIL config close.");
            return false;
        }
        if (!payload.isValid()) {
            logger.info("[PHOENIX-SNS] EMAIL message payload is empty.");
            return false;
        }
        return true;
    }

    private void sendEmail(String to, SnsPayload payload, RecipientMethod recipientMethod) {
        try {
            // 开启调试模式 && 收件人不在白名单 ==> 不发送邮件通知
            if (emailProperties.isDebugger() && !emailProperties.getWhiteList().contains(to)) {
                logger.info("[PHOENIX-SNS] EMAIL debugger: recipient userId({}) not in white list.", to);
                return;
            }
            logger.info("[PHOENIX-SNS] EMAIL message to userId={} sending...", to);
            var mailPayload = DefaultMailSaveParamVO.builder()
                    .content(payload.getContent())
                    .subject(payload.getSubject())
                    .htmlContent(true)
                    .from(emailProperties.getAdminId())
                    .recipientMethod(recipientMethod)
                    .to(to).build();
            logger.info("[PHOENIX-SNS] EMAIL payload: {}", mailPayload);
            mailer.send(mailPayload);
            logger.info("[PHOENIX-SNS] EMAIL message SENT.");
        } catch (Throwable e) {
            logger.error("[PHOENIX-SNS] EMAIL message sent FAILED.", e);
        }
    }
}
