package com.elitesland.cbpl.sns.notifier.service;

import cn.hutool.core.collection.CollUtil;
import com.elitesland.cbpl.sns.inbox.domain.InboxType;
import com.elitesland.cbpl.sns.inbox.service.InboxService;
import com.elitesland.cbpl.sns.mail.service.MailService;
import com.elitesland.cbpl.sns.notifier.domain.SnsChoice;
import com.elitesland.cbpl.sns.template.service.MsgTemplateService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.util.Set;

import static com.elitesland.cbpl.sns.notifier.domain.SnsTmplType.*;

/**
 * @author eric.hao
 * @since 2022/09/15
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class SnsServiceImpl implements SnsService {

    private final MsgTemplateService msgTemplateService;
    private final MailService mailService;
    private final InboxService inboxService;

    @Override
    public void sendMessage(InboxType inboxType, String msgCode, Set<String> tos, Object msgData) {
        if (CollUtil.isNotEmpty(tos)) {
            // 消息内容
            var message = todoMessage(msgCode, msgData);
            sendMessage(inboxType, tos, message);
        }
    }

    @Override
    public void sendExtraEmail(InboxType inboxType, String msgCode, Set<String> tos, Object msgData) {
        if (CollUtil.isNotEmpty(tos)) {
            var message = todoMessage(msgCode, msgData);
            mailService.sendByEmail(tos, message.emailOf());
        }
    }

    /**
     * 发送消息
     */
    private void sendMessage(InboxType inboxType, Set<String> tos, SnsChoice message) {
        logger.info("[PHOENIX-SNS] send message begin, to userId={}, content={}.", tos, message);
        // 站内信
        inboxService.sendInbox(inboxType, tos, message.inboxOf());
        // 发送邮件
        mailService.sendByUserId(tos, message.emailOf());
        // 发微信通知 TODO
        logger.info("[PHOENIX-SNS] send message finished.");
    }

    /**
     * 构造消息内容
     */
    private SnsChoice todoMessage(String msgCode, Object msgData) {
        var inboxTmpl = msgTemplateService.msgOf(msgCode, INBOX.getCode(), msgData);
        var emailTmpl = msgTemplateService.msgOf(msgCode, EMAIL.getCode(), msgData);
        var wechatTmpl = msgTemplateService.msgOf(msgCode, WECHAT.getCode(), msgData);
        return SnsChoice.of(inboxTmpl, emailTmpl, wechatTmpl);
    }
}
