package com.elitesland.cbpl.bpmn.service;

import cn.hutool.core.collection.CollUtil;
import com.elitesland.cbpl.bpmn.convert.BpmnInstanceConvert;
import com.elitesland.cbpl.bpmn.entity.BpmnInstanceDO;
import com.elitesland.cbpl.bpmn.provider.*;
import com.elitesland.cbpl.bpmn.udc.BpmnModule;
import com.elitesland.cbpl.bpmn.vo.param.BpmnParamVO;
import com.elitesland.cbpl.bpmn.vo.param.SensitiveWordParamVO;
import com.elitesland.cbpl.bpmn.vo.resp.BpmnCfgRespVO;
import com.elitesland.cbpl.bpmn.vo.resp.BpmnInstanceRespVO;
import com.elitesland.cbpl.bpmn.vo.resp.BpmnResultRespVO;
import com.elitesland.cbpl.iam.IamUserUtil;
import com.elitesland.yst.common.exception.BusinessException;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.util.Arrays;
import java.util.List;

/**
 * CBPL-BPMN 统一接口
 *
 * @author eric.hao
 * @since 2022/06/08
 */
@Slf4j
@Component
@RequiredArgsConstructor
public class BpmnService {

    private final BpmnCfgService bpmnCfgService;
    private final BpmnInstanceService bpmnInstanceService;
    private final JdeApprovalProvider jdeApprovalProvider;
    private final SapApprovalProvider sapApprovalProvider;
    private final QdaApprovalProvider qdaApprovalProvider;
    private final OaApprovalProvider oaApprovalProvider;
    private final SrmApprovalProvider srmApprovalProvider;
    private final LexiconApprovalProvider lexiconApprovalProvider;
    private final RoleApprovalProvider roleApprovalProvider;
    private final UserApprovalProvider userApprovalProvider;
    private final WorkflowApprovalProvider workflowApprovalProvider;
    private final B2bApprovalProvider b2bApprovalProvider;

    public BpmnResultRespVO startProcess(BpmnModule moduleKey, Long bizKey, SensitiveWordParamVO wordParamVO) {
        return startProcess(moduleKey, String.valueOf(bizKey), wordParamVO);
    }

    /**
     * 审批流程 - 发起
     */
    public BpmnResultRespVO startProcess(BpmnModule moduleKey, String bizKey, SensitiveWordParamVO wordParamVO) {
        BpmnCfgRespVO bpmnCfgRespVO = bpmnCfgService.bpmnCfgByModule(moduleKey.getCode());

        if (!bpmnCfgRespVO.isEnabled()) {
            logger.info("[CBPL-BPMN] {}的流程，无需审批。", moduleKey.getDesc());
            return new BpmnResultRespVO();
        }
        // 存在一个单据反复审批
//        BpmnInstanceRespVO instanceVO = bpmnInstanceService.bpmnInstanceByModuleBizKey(moduleKey, bizKey);
//        if (ObjectUtil.isNotNull(instanceVO)) {
//            throw new BusinessException("该单据已提交过审批，不能重复提交");
//        }

        // 对接外部系统
        if (bpmnCfgRespVO.isExternal()) {
            return switch (bpmnCfgRespVO.fromExternal()) {
                case JDE -> jdeApprovalProvider.startProcess(bpmnCfgRespVO, bizKey);
                case QDA -> qdaApprovalProvider.startProcess(bpmnCfgRespVO, bizKey);
                case OA -> oaApprovalProvider.startProcess(bpmnCfgRespVO, bizKey);
                case SAP -> sapApprovalProvider.startProcess(bpmnCfgRespVO, bizKey);
                case SRM -> srmApprovalProvider.startProcess(bpmnCfgRespVO, bizKey);
                case B2B -> b2bApprovalProvider.startProcess(bpmnCfgRespVO, bizKey);
            };
        }
        // 对接当前系统
        else if (bpmnCfgRespVO.isInternal()) {
            return switch (bpmnCfgRespVO.fromInternal()) {
                case AUTO -> lexiconApprovalProvider.startProcess(bpmnCfgRespVO, bizKey, wordParamVO);
                case ASSIGN_ROLE -> roleApprovalProvider.startProcess(bpmnCfgRespVO, bizKey);
                case ASSIGN_USER -> userApprovalProvider.startProcess(bpmnCfgRespVO, bizKey);
                case WORKFLOW -> workflowApprovalProvider.startProcess(bpmnCfgRespVO, bizKey);
            };
        }
        throw new BusinessException("[系统配置] 审批方式缺失，请检查");
    }

    public BpmnInstanceRespVO queryByBpmnInstId(Long bpmnInstId) {
        BpmnInstanceDO instanceDO = bpmnInstanceService.bpmnInstanceById(bpmnInstId);
        return BpmnInstanceConvert.INSTANCE.doToVO(instanceDO);
    }

    public BpmnResultRespVO agree(BpmnParamVO paramVO) {
        BpmnInstanceRespVO instanceVO = queryByBpmnInstId(paramVO.getBpmnInstId());
        List<String> userIds = Arrays.asList(instanceVO.getUserIds().split(","));
        String userIdStr = IamUserUtil.currentUserIdStr();
        if (CollUtil.isEmpty(userIds) || !userIds.contains(IamUserUtil.currentUserIdStr())) {
            throw new BusinessException("您没有权限审批该单据");
        }

        // TODO 当前状态是否可以审批同意

        BpmnCfgRespVO bpmnCfgRespVO = bpmnCfgService.bpmnCfgByModule(instanceVO.getModuleKey());
        // 对接外部系统
        if (bpmnCfgRespVO.isExternal()) {
            throw new BusinessException("已提交至" + bpmnCfgRespVO.fromExternal() + "系统审批");
        }
        // 对接当前系统
        else if (bpmnCfgRespVO.isInternal()) {
            return switch (bpmnCfgRespVO.fromInternal()) {
                case ASSIGN_ROLE -> roleApprovalProvider.agree(bpmnCfgRespVO, instanceVO);
                case ASSIGN_USER -> userApprovalProvider.agree(bpmnCfgRespVO, instanceVO);
                default -> {
                    String errorMsg = "[系统配置] " + bpmnCfgRespVO.fromInternal().getDesc() + "不支持该操作";
                    throw new BusinessException(errorMsg);
                }
            };
        }
        throw new BusinessException("[系统配置] 审批方式缺失，请检查");
    }

    public BpmnResultRespVO reject(BpmnParamVO paramVO) {
        BpmnInstanceRespVO instanceVO = queryByBpmnInstId(paramVO.getBpmnInstId());
        List<String> userIds = Arrays.asList(instanceVO.getUserIds().split(","));
        String userIdStr = IamUserUtil.currentUserIdStr();
        if (CollUtil.isEmpty(userIds) || !userIds.contains(IamUserUtil.currentUserIdStr())) {
            throw new BusinessException("您没有权限审批该单据");
        }

        // TODO 当前状态是否可以审批驳回

        BpmnCfgRespVO bpmnCfgRespVO = bpmnCfgService.bpmnCfgByModule(instanceVO.getModuleKey());
        // 对接外部系统
        if (bpmnCfgRespVO.isExternal()) {
            throw new BusinessException("已提交至" + bpmnCfgRespVO.fromExternal() + "系统审批");
        }
        // 对接当前系统
        else if (bpmnCfgRespVO.isInternal()) {
            return switch (bpmnCfgRespVO.fromInternal()) {
                case ASSIGN_ROLE -> roleApprovalProvider.reject(bpmnCfgRespVO, instanceVO);
                case ASSIGN_USER -> userApprovalProvider.reject(bpmnCfgRespVO, instanceVO);
                default -> {
                    String errorMsg = "[系统配置] " + bpmnCfgRespVO.fromInternal().getDesc() + "不支持该操作";
                    throw new BusinessException(errorMsg);
                }
            };
        }
        throw new BusinessException("[系统配置] 审批方式缺失，请检查");
    }
}
