package com.elitesland.cbpl.bpmn.service.impl;

import com.elitesland.cbpl.bpmn.convert.BpmnCfgConvert;
import com.elitesland.cbpl.bpmn.entity.BpmnCfgDO;
import com.elitesland.cbpl.bpmn.repo.BpmnCfgRepo;
import com.elitesland.cbpl.bpmn.repo.BpmnCfgRepoProc;
import com.elitesland.cbpl.bpmn.service.BpmnCfgService;
import com.elitesland.cbpl.bpmn.udc.*;
import com.elitesland.cbpl.bpmn.vo.param.BpmnCfgPageParamVO;
import com.elitesland.cbpl.bpmn.vo.param.BpmnCfgSaveParamVO;
import com.elitesland.cbpl.bpmn.vo.resp.BpmnCfgListRespVO;
import com.elitesland.cbpl.bpmn.vo.resp.BpmnCfgRespVO;
import com.elitesland.yst.common.base.PagingVO;
import com.elitesland.yst.common.base.param.OrderItem;
import com.elitesland.yst.common.exception.BusinessException;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.dubbo.common.utils.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author eric.hao
 * @since 2022/05/30
 */
@Slf4j
@Service
@AllArgsConstructor
public class BpmnCfgServiceImpl implements BpmnCfgService {

    private final BpmnCfgRepo bpmnCfgRepo;
    private final BpmnCfgRepoProc bpmnCfgRepoProc;

    @Override
    public PagingVO<BpmnCfgListRespVO> bpmnCfgPageBy(BpmnCfgPageParamVO paramVO) {
        long count = bpmnCfgRepoProc.bpmnCfgCountBy(paramVO);
        if (count > 0) {
            paramVO.setOrders(Collections.singletonList(new OrderItem("createTime", false)));
            var list = bpmnCfgRepoProc.bpmnCfgPageBy(paramVO);
            translateUdc(list);
            return new PagingVO<>(count, list);
        }
        return new PagingVO<>();
    }

    private void translateUdc(List<BpmnCfgListRespVO> list) {
        Map<String, List<BpmnModule>> moduleMap = Stream.of(BpmnModule.values()).collect(Collectors.groupingBy(BpmnModule::getCode));
        Map<String, List<BpmnEnv>> envMap = Stream.of(BpmnEnv.values()).collect(Collectors.groupingBy(BpmnEnv::getCode));
        Map<String, List<BpmnExternal>> externalMap = Stream.of(BpmnExternal.values()).collect(Collectors.groupingBy(BpmnExternal::getCode));
        Map<String, List<BpmnInternal>> internalMap = Stream.of(BpmnInternal.values()).collect(Collectors.groupingBy(BpmnInternal::getCode));
        Map<Integer, List<BpmnEnable>> enableMap = Stream.of(BpmnEnable.values()).collect(Collectors.groupingBy(BpmnEnable::getCode));
        if(CollectionUtils.isNotEmpty(list)){
            list.forEach(resp -> {
                resp.setModuleKeyName(StringUtils.isNotEmpty(resp.getModuleKey()) ? CollectionUtils.isNotEmpty(moduleMap.get(resp.getModuleKey())) ?
                        moduleMap.get(resp.getModuleKey()).get(0).getDesc() : "" : "");
                resp.setBpmnEnvName(StringUtils.isNotEmpty(resp.getBpmnEnv()) ? CollectionUtils.isNotEmpty(envMap.get(resp.getBpmnEnv())) ?
                        envMap.get(resp.getBpmnEnv()).get(0).getDesc() : "" : "");
                resp.setExternalName(StringUtils.isNotEmpty(resp.getExternal()) ? CollectionUtils.isNotEmpty(externalMap.get(resp.getExternal())) ?
                        externalMap.get(resp.getExternal()).get(0).getDesc() : "" : "");
                resp.setInternalName(StringUtils.isNotEmpty(resp.getInternal()) ? CollectionUtils.isNotEmpty(internalMap.get(resp.getInternal())) ?
                        internalMap.get(resp.getInternal()).get(0).getDesc() : "" : "");
                resp.setBpmnEnableName(resp.getBpmnEnable() != null ? CollectionUtils.isNotEmpty(enableMap.get(resp.getBpmnEnable())) ?
                        enableMap.get(resp.getBpmnEnable()).get(0).getDesc() : "" : "");
            });
        }
    }

    @Override
    public BpmnCfgRespVO bpmnCfgById(long bpmnCfgId) {
        Optional<BpmnCfgDO> bpmnCfgDO = bpmnCfgRepo.findById(bpmnCfgId);
        if (bpmnCfgDO.isEmpty()) {
            throw new BusinessException("[审批配置] " + bpmnCfgId + " 不存在");
        }
        return BpmnCfgConvert.INSTANCE.doToVO(bpmnCfgDO.get());
    }

    @Override
    public BpmnCfgRespVO bpmnCfgByModule(String moduleKey) {
        Optional<BpmnCfgDO> bpmnCfgDO = bpmnCfgRepo.findByModuleKey(moduleKey);
        if (bpmnCfgDO.isEmpty()) {
            throw new BusinessException("[审批配置] " + moduleKey + " 不存在");
        }
        return BpmnCfgConvert.INSTANCE.doToVO(bpmnCfgDO.get());
    }

    @Override
    public Long saveBpmnCfg(BpmnCfgSaveParamVO saveParam) {
        Optional<BpmnCfgDO> optional = bpmnCfgRepo.findByModuleKey(saveParam.getModuleKey());
        if(saveParam.getId() == null){
            if(optional.isPresent()){
                throw new BusinessException("[审批配置] " + saveParam.getModuleKey() + " 已存在，不能重复配置");
            }
            BpmnCfgDO bpmnCfgDO = BpmnCfgConvert.INSTANCE.saveParamToDo(saveParam);
            return bpmnCfgRepo.save(bpmnCfgDO).getId();
        } else {
            Optional<BpmnCfgDO> option = bpmnCfgRepo.findById(saveParam.getId());
            if(option.isEmpty()){
                throw new BusinessException("[审批配置] " + saveParam.getId() + " 不存在");
            }
            if(!saveParam.getModuleKey().equals(option.get().getModuleKey()) && optional.isPresent()){
                throw new BusinessException("[审批配置] " + saveParam.getModuleKey() + " 已存在，不能重复配置");
            }
            BpmnCfgDO bpmnCfgDO = BpmnCfgConvert.INSTANCE.saveParamToDo(saveParam);
            return bpmnCfgRepo.save(bpmnCfgDO).getId();
        }
    }

    @Override
    @Transactional
    public Integer deleteBpmnCfg(long id) {
        return bpmnCfgRepo.updateDeleteFlag(id, 1);
    }
}
