package com.elitesland.cbpl.kumiho.aop;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitesland.cbpl.elastic.repository.ElasticRepository;
import com.elitesland.cbpl.kumiho.handler.KumihoHandler;
import com.elitesland.cbpl.kumiho.util.KumihoEsUtil;
import com.elitesland.cbpl.iam.IamUserUtil;
import com.elitesland.cbpl.kumiho.annotation.Kumiho;
import com.elitesland.cbpl.unicom.util.ApplicationContextGetBeanHelper;
import com.elitesland.cbpl.util.ExceptionUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.AfterReturning;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.task.TaskExecutor;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.concurrent.CompletableFuture;

/**
 * @author eric.hao
 * @since 2022/10/10
 */
@Aspect
@Slf4j
@Component
@RequiredArgsConstructor
public class KumihoCommandBus {

    /**
     * 使用CloudTi架构提供的线程池
     * {@link com.elitesland.cloudt.context.threadpool.ThreadPoolAutoConfiguration}
     */
    @Autowired
    private TaskExecutor taskExecutor;
    @Autowired
    private ElasticRepository elasticRepository;

    @Pointcut("@annotation(com.elitesland.cbpl.kumiho.annotation.Kumiho)")
    public void command() {
    }

    @AfterReturning(value = "command()", returning = "retObj")
    public void after(JoinPoint point, Object retObj) {
        CompletableFuture.runAsync(() -> {
            // 读取类实例
            Class<? extends KumihoHandler>[] clazzArr = ((MethodSignature) point.getSignature()).getMethod().getAnnotation(Kumiho.class).value();
            for (var clazz : clazzArr) {
                var handler = (KumihoHandler) ApplicationContextGetBeanHelper.getBean(clazz.getName());
                if (ObjectUtil.isNull(handler)) {
                    throw new RuntimeException("[KUMIHO] " + clazz.getName() + "：未找到类实例");
                }
                // 读到待同步的数据
                List<?> data = handler.aggregate(point.getArgs(), retObj);
                String tenantId = IamUserUtil.currentTenantIdStr();
                logger.debug("[KUMIHO] 租户({}), 同步的实体类({}), 待同步数据：{}", tenantId, clazz.getName(), data);
                dataAsync(data);
            }
        }, taskExecutor).whenComplete((resp, ex) -> {
            if (ObjectUtil.isNotNull(ex)) {
                logger.error(ExceptionUtil.formatException(ex));
            }
        });
    }

    private void dataAsync(List<?> data) {
        if (CollUtil.isEmpty(data)) {
            return;
        }
        Class<?> clazz = data.get(0).getClass();
        // 分批次同步到ES
        elasticRepository.batchInsert(data, KumihoEsUtil.indexNameByClz(clazz));
    }
}
