package com.elitesland.cbpl.rosefinch.service;

import cn.hutool.core.collection.CollUtil;
import com.elitesland.cbpl.rosefinch.entity.RosefinchDetailDO;
import com.elitesland.cbpl.rosefinch.entity.RosefinchLogDO;
import com.elitesland.cbpl.rosefinch.udc.LogStatusEnum;
import com.elitesland.yst.common.base.ApiCode;
import com.elitesland.yst.common.exception.BusinessException;
import com.elitesland.yst.system.service.SysNumberRuleService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.apache.commons.lang3.StringUtils;
import org.apache.dubbo.config.annotation.DubboReference;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.task.TaskExecutor;
import org.springframework.stereotype.Component;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;

/**
 * @author eric.hao
 * @since 2022/06/20
 */
@Slf4j
@Component
@RequiredArgsConstructor
public class RosefinchExecutor {

    /**
     * 使用CloudTi架构提供的线程池
     * {@link com.elitesland.cloudt.context.threadpool.ThreadPoolAutoConfiguration}
     */
    @Autowired
    private TaskExecutor taskExecutor;
    private final RosefinchService rosefinchService;

    /**
     * 异步批处理
     *
     * @param asyncType 处理类型
     * @param source    源数据
     * @param func      行处理函数
     * @param <T>       对象类型
     */
    public <T> void runAsync(String asyncType, List<T> source, Function<T, String> func) {
        // 记录初始批处理日志
        RosefinchLogDO headLog = rosefinchService.init(asyncType, source.size());
        headLog.setTotalCount(source.size());

        List<RosefinchDetailDO> detailLogList = new ArrayList<>();
        // 空数据
        if (CollUtil.isEmpty(source)) {
            headLog.setEndTime(LocalDateTime.now());
            rosefinchService.save(headLog, detailLogList);
            return;
        }

        // 异步执行
        AtomicInteger successCnt = new AtomicInteger();
        AtomicInteger failCnt = new AtomicInteger();
        CompletableFuture<Void> cf = CompletableFuture.runAsync(() -> {
            for (int i = 0; i < source.size(); i++) {
                RosefinchDetailDO detailDO = new RosefinchDetailDO();
                detailDO.setLineNo(i + 1);
                detailDO.setStartTime(LocalDateTime.now());
                try {
                    detailDO.setLogStatus("S");
                    detailDO.setErrorMsg(func.apply(source.get(i)));
                    detailLogList.add(detailDO);
                    successCnt.getAndIncrement();
                } catch (Exception e) {
                    detailDO.setLogStatus("E");
                    detailDO.setErrorMsg(e.getMessage());
                    detailLogList.add(detailDO);
                    failCnt.getAndIncrement();
                }
                detailDO.setEndTime(LocalDateTime.now());
            }
            headLog.setSuccessCount(successCnt.get());
            headLog.setFailCount(failCnt.get());
        }, taskExecutor);

        // 成功结束
        cf.thenAccept((result) -> {
            headLog.setLogStatus(LogStatusEnum.TRUE.getValue());
            headLog.setEndTime(LocalDateTime.now());
            rosefinchService.save(headLog, detailLogList);
        });
    }
}
