package com.elitesland.cbpl.tool.version.util;

import com.elitesland.cbpl.tool.version.annotation.PhoenixVersion;
import com.elitesland.cbpl.tool.version.domain.VersionVO;
import lombok.SneakyThrows;
import org.reflections.Reflections;
import org.reflections.util.ClasspathHelper;
import org.reflections.util.ConfigurationBuilder;

import java.io.InputStream;
import java.net.URL;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

import static com.elitesland.cbpl.tool.version.constant.GitAttr.GIT_COMMIT_MODULE;
import static com.elitesland.cbpl.tool.version.constant.GitAttr.GIT_PROPERTIES;
import static org.reflections.scanners.Scanners.SubTypes;
import static org.reflections.scanners.Scanners.TypesAnnotated;

/**
 * @author eric.hao
 * @since 2024/02/29
 */
public class PhoenixVersionUtil {

    /**
     * 获取所有 Phoenix 工具集的版本
     *
     * @return 版本信息
     */
    public static List<VersionVO> dependencies(List<String> basePackages) {
        return dependencies(reflections(basePackages));
    }

    public static List<VersionVO> dependencies(Reflections reflections) {
        // 遍历声明的工具版本
        Set<Class<?>> typesAnnotatedWith = reflections.getTypesAnnotatedWith(PhoenixVersion.class);
        return typesAnnotatedWith.stream().map(row ->
                ManifestUtils.dependency(row, loadProperties())
        ).collect(Collectors.toList());
    }

    /**
     * 读取所有jar包的git信息
     *
     * @return git.properties
     */
    @SneakyThrows
    private static Map<String, Properties> loadProperties() {
        List<Properties> result = new ArrayList<>();
        ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
        Enumeration<URL> enums = classLoader.getResources(GIT_PROPERTIES);
        while (enums.hasMoreElements()) {
            URL tme = enums.nextElement();
            // 通过UrlConnection获取输入流
            InputStream in = tme.openConnection().getInputStream();
            // 将输入流转化为Properties类
            Properties properties = new Properties();
            properties.load(in);
            result.add(properties);
        }
        return result.stream().collect(Collectors.toMap(row ->
                (String) row.get(GIT_COMMIT_MODULE), Function.identity()));
    }

    /**
     * 反射类扫描注册
     */
    private static Reflections reflections(List<String> basePackages) {
        Collection<URL> urls = new ArrayList<>();
        for (String basePackage : basePackages) {
            urls.addAll(ClasspathHelper.forPackage(basePackage));
        }
        return new Reflections(new ConfigurationBuilder().setUrls(urls).addScanners(TypesAnnotated, SubTypes));
    }
}
