package com.elitesland.cbpl.tool.mongo.repository;

import com.elitesland.cbpl.tool.db.PageQueryParam;
import com.elitesland.cbpl.tool.db.PagingVO;
import com.elitesland.cbpl.tool.mongo.domain.MongoDoc;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Query;

import java.util.List;

/**
 * MongoDB 查询条件
 *
 * @author eric.hao
 * @since 2022/09/15
 */
@Slf4j
public abstract class MongoQuery extends PageQueryParam {

    /**
     * @return 集合名
     */
    public abstract String getStoreName();

    /**
     * 构建查询条件
     *
     * @return 查询条件
     */
    protected abstract Query buildQuery();

    /**
     * 查询(for paging)
     *
     * @param template    Mongo API 实例
     * @param entityClass 记录实体类型
     * @return 分页数据集
     */
    public <T extends MongoDoc> PagingVO<T> paging(MongoTemplate template, Class<T> entityClass) {
        final Query query = buildQuery();
        final long total = template.count(query, getStoreName());
        if (total > 0) {
            final List<T> rows = template.find(withPaging(query), entityClass, getStoreName());
            return new PagingVO<>(total, rows);
        }
        return new PagingVO<>();
    }

    /**
     * 统计数据量
     *
     * @param template Mongo API 实例
     * @return 数据量
     */
    public long count(MongoTemplate template) {
        return template.count(buildQuery(), getStoreName());
    }

    /**
     * 查询
     *
     * @param template    Mongo API 实例
     * @param entityClass 记录实体类型
     * @return 行数据集
     */
    public <T> List<T> find(MongoTemplate template, Class<T> entityClass) {
        return template.find(withSorting(buildQuery()), entityClass, getStoreName());
    }

    /**
     * @param query 查询条件
     * @return 加上分页排序的查询条件
     */
    private Query withPaging(Query query) {
        if (getPageNumber() > 0L) {
            int offset = getPageNumber() * getSize();
            query.skip(offset);
        }
        if (getSize() > 0L) {
            query.limit(getSize());
        }
        return withSorting(query);
    }

    /**
     * @param query 查询条件
     * @return 加上排序的查询条件
     */
    private Query withSorting(Query query) {
        // 默认事件排序
        Sort timeSort = Sort.by(Sort.Direction.DESC, "createTime");
        query.with(timeSort);
        logger.info("[MONGO] query: {}", query);
        return query;
    }

}
