package com.elitesland.cbpl.tool.tenant;

import cn.hutool.core.text.StrPool;
import cn.hutool.core.util.StrUtil;
import cn.hutool.extra.spring.SpringUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;

import java.util.function.Supplier;

/**
 * @author eric.hao
 * @since 2024/01/16
 */
@Slf4j
public class TenantSpiUtil {

    /**
     * 设置当前租户Session
     *
     * @param tenantId 租户相关配置
     * @since 0.1.15-SNAPSHOT
     */
    public static void setCurrentTenant(Long tenantId) {
        try {
            TenantClientSpi tenantClient = SpringUtil.getBean(TenantClientSpi.class);
            tenantClient.setCurrentTenant(tenantId);
        } catch (NoSuchBeanDefinitionException e) {
            logger.warn("[PHOENIX-TENANT] TenantClientSpi NoSuchBeanDefinition");
        }
    }

    /**
     * 设置当前租户Session
     *
     * @param tenantCode 租户相关配置
     * @since 0.1.15-SNAPSHOT
     */
    public static void setCurrentTenant(String tenantCode) {
        try {
            TenantClientSpi tenantClient = SpringUtil.getBean(TenantClientSpi.class);
            tenantClient.setCurrentTenant(tenantCode);
        } catch (NoSuchBeanDefinitionException e) {
            logger.warn("[PHOENIX-TENANT] TenantClientSpi NoSuchBeanDefinition");
        }
    }

    /**
     * 清除当前租户的Session
     *
     * @since 0.1.15-SNAPSHOT
     */
    public static void resetCurrentTenant() {
        try {
            TenantClientSpi tenantClient = SpringUtil.getBean(TenantClientSpi.class);
            tenantClient.resetCurrentTenant();
        } catch (NoSuchBeanDefinitionException e) {
            logger.warn("[PHOENIX-TENANT] TenantClientSpi NoSuchBeanDefinition");
        }
    }

    /**
     * 当前租户编码
     *
     * @return 租户编码
     */
    public static String currentTenantCode() {
        try {
            TenantClientSpi tenantClient = SpringUtil.getBean(TenantClientSpi.class);
            return tenantClient.currentTenantCode();
        } catch (NoSuchBeanDefinitionException e) {
            return "";
        }
    }

    /**
     * 租户前缀 (tenantCode_)
     *
     * @return 前缀
     */
    public static String tenantPrefix() {
        return tenantPrefix(StrPool.C_UNDERLINE);
    }

    /**
     * 租户前缀 (tenantCode_)
     *
     * @return 前缀
     */
    public static String tenantPrefix(char prefix) {
        String tenantCode = currentTenantCode();
        return StrUtil.isBlank(tenantCode) ? "" : tenantCode + prefix;
    }

    /**
     * 所有租户下都执行以下方法
     *
     * @param executor 业务代码
     */
    public static void byTenants(Runnable executor) {
        try {
            TenantClientSpi tenantClient = SpringUtil.getBean(TenantClientSpi.class);
            tenantClient.byTenants(executor);
        }
        // 未配置租户实现，直接执行
        catch (NoSuchBeanDefinitionException e) {
            logger.warn("[PHOENIX-TENANT] TenantClientSpi NoSuchBeanDefinition");
            executor.run();
        }
    }

    /**
     * 直接使用指定的租户
     * <p>
     * 以Runnable函数式接口类型为参数，没有返回结果
     *
     * @param executor   业务侧执行方法
     * @param tenantCode 租户编码
     * @since 0.1.13-SNAPSHOT
     */
    public static void byTenantDirectly(Runnable executor, String tenantCode) {
        try {
            TenantClientSpi tenantClient = SpringUtil.getBean(TenantClientSpi.class);
            tenantClient.byTenantDirectly(executor, tenantCode);
        }
        // 未配置租户实现，直接执行
        catch (NoSuchBeanDefinitionException e) {
            logger.warn("[PHOENIX-TENANT] TenantClientSpi NoSuchBeanDefinition");
            executor.run();
        }
    }

    /**
     * 直接使用指定的租户
     * <p>
     * 以Supplier函数式接口类型为参数，返回结果类型为E
     *
     * @param executor   业务操作
     * @param tenantCode 租户编码
     * @param <E>        数据类型
     * @return 数据
     * @since 0.1.13-SNAPSHOT
     */
    public static <E> E byTenantDirectly(Supplier<E> executor, String tenantCode) {
        try {
            TenantClientSpi tenantClient = SpringUtil.getBean(TenantClientSpi.class);
            return tenantClient.byTenantDirectly(executor, tenantCode);
        }
        // 未配置租户实现，直接执行
        catch (NoSuchBeanDefinitionException e) {
            logger.warn("[PHOENIX-TENANT] TenantClientSpi NoSuchBeanDefinition");
            return executor.get();
        }
    }
}
