package com.elitesland.srm.iam.service;

import cn.hutool.core.util.ObjectUtil;
import com.elitesland.cbpl.unicom.annotation.UnicomTag;
import com.elitesland.srm.supplier.biz.vo.save.SuppSaveParam;
import com.elitesland.srm.util.enums.UserRoleType;
import com.elitesland.srm.util.enums.UserType;
import com.elitesland.yst.common.base.ApiCode;
import com.elitesland.yst.common.base.ApiResult;
import com.elitesland.yst.common.constant.CommonConstant;
import com.elitesland.yst.common.constant.Terminal;
import com.elitesland.yst.common.exception.BusinessException;
import com.elitesland.yst.core.common.constants.YstSetting;
import com.elitesland.yst.system.provider.SysRoleRpcService;
import com.elitesland.yst.system.provider.SysUserRpcService;
import com.elitesland.yst.system.provider.dto.SysRoleRpcDTO;
import com.elitesland.yst.system.provider.dto.SysUserRpcDTO;
import com.elitesland.yst.system.provider.param.SysRoleRpcDtoParam;
import com.elitesland.yst.system.provider.param.SysUserRpcDtoParam;
import com.elitesland.yst.system.provider.param.SysUserRpcSaveParam;
import com.elitesland.yst.system.service.SysSettingService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.dubbo.config.annotation.DubboReference;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @author eric.hao
 * @since 2023/01/03
 */
@Slf4j
@Service
@RequiredArgsConstructor
@UnicomTag("GENERAL")
public class IamUserServiceImpl implements IamUserService<SuppSaveParam> {

    @DubboReference
    private SysUserRpcService userRpcService;
    @DubboReference
    private SysRoleRpcService roleRpcService;
    @DubboReference(version = "${provider.service.version}")
    private SysSettingService sysSettingService;

    @Override
    public Long createSuppUser(SuppSaveParam saveParam) {
        SysUserRpcSaveParam userParam = new SysUserRpcSaveParam();
        userParam.setMobile(StringUtils.defaultIfBlank(saveParam.getMobile(), saveParam.getSuppCode()));
        userParam.setEmail(saveParam.getEmail());
        userParam.setUsername(saveParam.getSuppCode());
        userParam.setFirstName(saveParam.getSuppName());
        userParam.setRoleIds(queryRoleIds(UserRoleType.POTENTIAL.getCode()));
        userParam.setUserType(UserType.SUPP.getCode());
        userParam.setPassword(saveParam.getPassword());
        return createUser(userParam, true);
    }

    /**
     * 创建账号
     *
     * @param userParam 保存参数
     * @param enabled   是否启用
     * @return 用户ID
     */
    private Long createUser(SysUserRpcSaveParam userParam, boolean enabled) {
        userParam.setMobile(getUniqueMobile(userParam.getMobile()));
        userParam.setEmail(getUniqueEmail(userParam.getEmail()));
        userParam.setEnabled(enabled);
        userParam.setSourceType("OTH");
        userParam.setPassword(StringUtils.defaultIfBlank(userParam.getPassword(), generateDefaultPwd()));
        Set<Terminal> terminals = new HashSet<>();
        terminals.add(Terminal.BACKEND);
        terminals.add(Terminal.APP);
        terminals.add(Terminal.APPLET);
        userParam.setTerminals(terminals);
        ApiResult<Object> apiResult = userRpcService.sysUserAccountCreate(userParam);
        return Long.valueOf(apiResult.getData().toString());
    }

    /**
     * 获取买家角色id
     */
    private List<Long> queryRoleIds(String roleCode) {
        List<String> roleCodes = new ArrayList<>();
        roleCodes.add(roleCode);
        SysRoleRpcDtoParam param = new SysRoleRpcDtoParam();
        param.setRoleCodes(roleCodes);
        List<SysRoleRpcDTO> roles = roleRpcService.findRoleRpcDtoByParam(param);
        return roles.stream().map(SysRoleRpcDTO::getId).collect(Collectors.toList());
    }

    /**
     * 获取唯一手机号：重复时则追加#1
     *
     * @param mobile 手机号
     * @return 手机号#size
     */
    private String getUniqueMobile(String mobile) {
        SysUserRpcDtoParam param = new SysUserRpcDtoParam();
        param.setRepeatMobile(mobile);
        List<SysUserRpcDTO> sysUserDTO = userRpcService.findUserRpcDtoByParam(param);
        int size = sysUserDTO.size();
        return size > 0 ? String.format("%s#%d", mobile, size) : mobile;
    }

    /**
     * 获取唯一邮箱：重复时则追加#1
     *
     * @param email 手机号
     * @return 邮箱#size
     */
    private String getUniqueEmail(String email) {
        SysUserRpcDtoParam param = new SysUserRpcDtoParam();
        param.setRepeatEmail(email);
        List<SysUserRpcDTO> sysUserDTO = userRpcService.findUserRpcDtoByParam(param);
        int size = sysUserDTO.size();
        return size > 0 ? String.format("%s#%d", email, size) : email;
    }

    /**
     * 默认密码
     */
    private String generateDefaultPwd() {
        String newPassword;
        var defaultPwdSettings = sysSettingService.oneByNo(YstSetting.SYS_DEFAULT_USER_PWD);
        if (ObjectUtil.isNull(defaultPwdSettings)) {
            // 如果没有密码配置，则取123456为默认密码
            newPassword = CommonConstant.INIT_PWD;
        } else {
            newPassword = defaultPwdSettings.getSettingVal();
            if (StringUtils.isBlank(newPassword)) {
                throw new BusinessException(ApiCode.BUSINESS_EXCEPTION, "系统配置的默认密码为空，请检查系统配置是否正确");
            }
        }
        return newPassword;
    }
}
