package com.elitesland.yst.common.exception.handler;

import cn.hutool.core.exceptions.ExceptionUtil;
import cn.hutool.core.util.StrUtil;
import com.elitesland.yst.common.base.ApiCode;
import com.elitesland.yst.common.base.ApiResult;
import com.elitesland.yst.common.common.CloudtContextHolder;
import com.elitesland.yst.common.exception.BadCaptchaException;
import com.elitesland.yst.common.exception.BadJwtTokenException;
import com.elitesland.yst.common.exception.BusinessException;
import com.elitesland.yst.common.util.ThrowableUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.dubbo.remoting.TimeoutException;
import org.apache.dubbo.rpc.RpcException;
import org.springframework.dao.DataAccessResourceFailureException;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.client.HttpClientErrorException;
import org.springframework.web.client.ResourceAccessException;

import javax.validation.ConstraintViolation;
import javax.validation.ConstraintViolationException;
import javax.validation.ValidationException;
import javax.ws.rs.BadRequestException;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;

/**
 * <pre>
 * [功能说明]
 * </pre>
 *
 * @author Mir
 * @date 2020/6/22
 */
@Slf4j
@ControllerAdvice
public class GlobalExceptionHandler {

    @ExceptionHandler(BadRequestException.class)
    @ResponseBody
    public ApiResult<Object> handleBadRequestException(BadRequestException e) {
        String errorNo = printError(e, "请求错误");
        return ApiResult.fail(ApiCode.SYSTEM_EXCEPTION, errorNo, ThrowableUtil.getStackTrace(e), null);
    }

    @ExceptionHandler(HttpMessageNotReadableException.class)
    @ResponseBody
    public ApiResult<Object> handleHttpMessageNotReadableException(HttpMessageNotReadableException e) {
        String errorNo = printError(e, "请求数据格式有误");
        return ApiResult.fail(ApiCode.PARAMETER_PARSE_EXCEPTION, errorNo, ThrowableUtil.getStackTrace(e), null);
    }

    @ExceptionHandler(HttpRequestMethodNotSupportedException.class)
    @ResponseBody
    public ApiResult<Object> handleHttpRequestMethodNotSupportedException(HttpRequestMethodNotSupportedException e) {
        String errorNo = printError(e, "请求方式有误");
        return ApiResult.fail(ApiCode.METHOD_NOT_SUPPORT, errorNo, ThrowableUtil.getStackTrace(e), null);
    }

    @ExceptionHandler(BusinessException.class)
    @ResponseBody
    public ApiResult<Object> handleBusinessException(BusinessException e) {
        String errorNo = printError(e, "业务异常");
        return ApiResult.fail(e.getCode() == null ? ApiCode.BUSINESS_EXCEPTION : e.getCode(), errorNo, ThrowableUtil.getStackTrace(e), e.getMessage());
    }

    @ExceptionHandler(BadCaptchaException.class)
    @ResponseBody
    public ApiResult<Object> handleBadCaptchaException(BadCaptchaException e) {
        String errorNo = printError(e, "验证码验证失败");
        return ApiResult.fail(ApiCode.VERIFICATION_CODE_EXCEPTION, errorNo, "验证码验证失败", e.getMessage());
    }

    @ExceptionHandler(BadJwtTokenException.class)
    @ResponseBody
    public ApiResult<Object> handleAuthenticationException(BadJwtTokenException e) {
        String errorNo = printError(e, "权限认证失败");
        return ApiResult.fail(ApiCode.UNAUTHENTICATED_EXCEPTION, errorNo, ThrowableUtil.getStackTrace(e), null);
    }

    @ExceptionHandler(ResourceAccessException.class)
    @ResponseBody
    public ApiResult<Object> handleAuthenticationException(ResourceAccessException e) {
        String errorNo = printError(e, "请求资源异常");
        return ApiResult.fail(ApiCode.SYSTEM_EXCEPTION, errorNo, ThrowableUtil.getStackTrace(e), null);
    }

    @ExceptionHandler(HttpClientErrorException.class)
    @ResponseBody
    public ApiResult<Object> handleHttpClientErrorException(HttpClientErrorException e) {
        String errorNo = printError(e, "认证授权异常");

        return ApiResult.fail(ApiCode.AUTHENTICATION_EXCEPTION, errorNo, ThrowableUtil.getStackTrace(e), null);
    }

    @ExceptionHandler({Exception.class})
    @ResponseBody
    public ApiResult<Object> handleException(Exception e) {
        String errorNo = printError(e, "系统异常");

        return ApiResult.fail(ApiCode.SYSTEM_EXCEPTION, errorNo, ThrowableUtil.getStackTrace(e), null);
    }

    /**
     * 参数校验异常
     *
     * @param exception 异常
     * @return 结果
     */
    @ExceptionHandler({IllegalArgumentException.class, MethodArgumentNotValidException.class,
            ConstraintViolationException.class, ValidationException.class})
    @ResponseBody
    public ApiResult<Object> handleArgumentInvalidException(Exception exception) {
        String errorNo = printError(exception, "参数校验不通过");

        if (exception instanceof IllegalArgumentException) {
            return ApiResult.fail(ApiCode.PARAMETER_EXCEPTION, errorNo, null, exception.getMessage());
        }

        if (exception instanceof MethodArgumentNotValidException) {
            String msg = ((MethodArgumentNotValidException) exception).getBindingResult().getAllErrors().get(0).getDefaultMessage();
            return ApiResult.fail(ApiCode.PARAMETER_EXCEPTION, errorNo, null, msg);
        }

        if (exception instanceof ConstraintViolationException) {
            String msg = ((ConstraintViolationException) exception).getConstraintViolations().toArray(ConstraintViolation[]::new)[0].getMessage();
            return ApiResult.fail(ApiCode.PARAMETER_EXCEPTION, errorNo, null, msg);
        }

        if (exception instanceof ValidationException) {
            return ApiResult.fail(ApiCode.PARAMETER_EXCEPTION, errorNo, null, exception.getMessage());
        }

        return ApiResult.fail(ApiCode.PARAMETER_EXCEPTION, errorNo, null, ExceptionUtil.getRootCause(exception).getMessage());
    }

    private static final String EXP_NO_PROVIDER_AVAILABLE = "No provider available";
    private static final String EXP_NO_PROVIDER_AVAILABLE1 = "because channel is closed";
    private static final String EXP_THREAD_POOL_OVERFLOW = "thread pool is exhausted";

    @ExceptionHandler(RpcException.class)
    @ResponseBody
    public ApiResult<Object> handleRpcException(RpcException exception) {
        String errorNo = printError(exception, "调用远程接口失败");

        var rootCause = ExceptionUtil.getRootCause(exception);
        if (rootCause instanceof TimeoutException) {
            return ApiResult.fail(ApiCode.SERVER_RESPONSE_TIMEOUT, errorNo, ThrowableUtil.getStackTrace(exception), "请求超时");
        }

        var rootCauseMsg = rootCause.getMessage();
        if (StrUtil.contains(rootCauseMsg, EXP_NO_PROVIDER_AVAILABLE) || StrUtil.contains(rootCauseMsg, EXP_NO_PROVIDER_AVAILABLE1)) {
            return ApiResult.fail(ApiCode.CLOUD_SERVICE_UNAVAILABLE, errorNo, ThrowableUtil.getStackTrace(exception), "系统维护中，请稍后再试");
        }
        if (StrUtil.contains(rootCauseMsg, EXP_THREAD_POOL_OVERFLOW)) {
            return ApiResult.fail(ApiCode.THREAD_OVERFLOW, errorNo, ThrowableUtil.getStackTrace(exception), "当前访问用户过多，请稍后再试");
        }
        return ApiResult.fail(ApiCode.CLOUD_SERVICE_EXCEPTION, errorNo, ThrowableUtil.getStackTrace(exception), null);
    }

    @ExceptionHandler(DataAccessResourceFailureException.class)
    @ResponseBody
    public ApiResult<Object> handleDataAccessException(DataAccessResourceFailureException exception) {
        String errorNo = printError(exception, "数据库处理异常");

        return ApiResult.fail(ApiCode.DAO_EXCEPTION, errorNo, ThrowableUtil.getStackTrace(exception), CloudtContextHolder.getGlobalDefaultErrorMsg());
    }

    private static final DateTimeFormatter FORMATTER = DateTimeFormatter.ofPattern("yyyyMMddHHmmssSSS");
    private String printError(Throwable throwable, String description) {
        String errorNo = FORMATTER.format(LocalDateTime.now());
        String msg = description + "，错误号：" + errorNo;
        log.error(msg, throwable);

        return errorNo;
    }
}
