package com.elitesland.tw.tw5.api.bpm.common.util;

import cn.hutool.core.date.DatePattern;
import lombok.extern.slf4j.Slf4j;

import java.lang.reflect.Field;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Map;

/**
 * 特殊类型字段 统一处理 Boolean DateTime 类型转换工具类
 * 用于将实体类中的Boolean类型字段转换为"是"/"否"字符串表示
 * DateTime 格式化为 "yyyy-MM-dd HH:mm:ss"
 * 支持处理继承层次结构中的字段
 */
@Slf4j
public class SpecConverterUtil {

    /**
     * 将对象中所有的Boolean类型字段转换为"是"/"否"并放入目标Map中
     * <p>
     * DateTime 格式化为 "yyyy-MM-dd HH:mm:ss"
     * <p>
     * <p>
     * 包括父类中的Boolean字段
     *
     * @param sourceObj 源对象
     * @param targetMap 目标Map
     */
    public static void convertSpecFieldsToString(Object sourceObj, Map<String, Object> targetMap) {
        if (sourceObj == null || targetMap == null) {
            return;
        }

        // 递归处理当前类及其父类的所有字段
        Class<?> clazz = sourceObj.getClass();
        while (clazz != null) {
            Field[] fields = clazz.getDeclaredFields();
            for (Field field : fields) {
                processSpecField(sourceObj, targetMap, field);
            }
            // 移动到父类
            clazz = clazz.getSuperclass();
        }
    }

    /**
     * 处理单个Boolean字段
     * <p>
     * DateTime 格式化为 "yyyy-MM-dd HH:mm:ss"
     *
     * @param sourceObj 源对象
     * @param targetMap 目标Map
     * @param field     字段
     */
    private static void processSpecField(Object sourceObj, Map<String, Object> targetMap, Field field) {
        if (field.getType().equals(Boolean.class) || field.getType().equals(boolean.class)) {
            try {
                field.setAccessible(true);
                Object value = field.get(sourceObj);
                if (value != null) {
                    targetMap.put(field.getName(), (Boolean) value ? "是" : "否");
                }
            } catch (IllegalAccessException e) {
                // 忽略访问异常，继续处理其他字段
                log.error("processSpecField处理异常", e);
            }
        }
        if (field.getType().equals(LocalDateTime.class)) {
            try {
                field.setAccessible(true);
                Object value = field.get(sourceObj);
                if (value != null) {
                    targetMap.put(field.getName(), ((LocalDateTime) value).format(DateTimeFormatter.ofPattern(DatePattern.NORM_DATETIME_PATTERN)));
                }
            } catch (IllegalAccessException e) {
                // 忽略访问异常，继续处理其他字段
                log.error("processSpecField处理异常", e);
            }
        }
    }
}
